///////////////////////////////////////////////////////////////////////////////
//                                                                           //
//                      OllyGraph Plugin for OllyDbg                         //
//                                                                           //
//                     Copyright (C) 2004 Joe Stewart                        //
//                                                              	     //
// This plugin allows you to generate VCG-compatible function graphs much    //
// like the same functionality in IDA Pro.                                   //
//                                                                           //
// This code is distributed "as is", without warranty of any kind, expressed //
// or implied, including, but not limited to warranty of fitness for any     //
// particular purpose. In no event will Joe Stewart be liable to you for any //
// special, incidental, indirect, consequential or any other damages caused  //
// by the use, misuse, or the inability to use of this code, including any   //
// lost profits or lost savings, even if Joe Stewart has been advised of the //
// possibility of such damages. Or, translated into English: use at your own //
// risk!                                                                     //
//                                                                           //
// This code is free. You can modify this code, include parts of it in your  //
// own programs and redistribute modified code provided that you remove all  //
// copyright messages or, if changes are significant enough, substitute them //
// with your own copyright.                                                  //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

// VERY IMPORTANT NOTICE: COMPILE THIS DLL WITH BYTE ALIGNMENT OF STRUCTURES
// AND UNSIGNED CHAR!

#include <windows.h>
#include <stdio.h>
#include <string.h>

#include "plugin.h"

typedef struct t_disasm_list_element {
	t_disasm disasm;
	void *next_element;
} t_disasm_list_element;

typedef struct previous_edge_struct {
	ulong source;
	ulong targetfalse;
	ulong targettrue;
} previous_edge_struct;

HINSTANCE        hinst;
HWND             hwmain;

#define MAX_EDGELIST_SIZE 256000
#define MAX_NODES 512

ulong nodelist[MAX_NODES];
int nodes;

void GenerateFunctionFlowchart(void);
void GenerateFunctionCallGraph(void);
void GenerateXRefsToAddressGraph(void); // not implemented
void GenerateXRefsFromAddressGraph(void); // not implemented

static void free_list(void *t_disasm_list);
static void append_nodelist(ulong addr);
static int nodestart(ulong addr);
static void sanitize(char *comment, char *cleaned); 
static void cleanup_tempfiles(char *tmppath);

char vcg_params[]="manhattan_edges: yes\n"
"layoutalgorithm: mindepth\n"
"finetuning: no\n"
"layout_downfactor: 100\n"
"layout_upfactor: 0\n"
"layout_nearfactor: 0\n"
"xlspace: 12\n"
"yspace: 30\n"
"colorentry 32: 0 0 0\n"
"colorentry 33: 0 0 255\n"
"colorentry 34: 0 0 255\n"
"colorentry 35: 128 128 128\n"
"colorentry 36: 128 128 128\n"
"colorentry 37: 0 0 128\n"
"colorentry 38: 0 0 128\n"
"colorentry 39: 0 0 255\n"
"colorentry 40: 0 0 255\n"
"colorentry 41: 0 0 128\n"
"colorentry 42: 0 128 0\n"
"colorentry 43: 0 255 0\n"
"colorentry 44: 0 128 0\n"
"colorentry 45: 255 128 0\n"
"colorentry 46: 0 128 0\n"
"colorentry 47: 128 128 255\n"
"colorentry 48: 255 0 0\n"
"colorentry 49: 128 128 0\n"
"colorentry 50: 1 1 1\n"
"colorentry 51: 192 192 192\n"
"colorentry 52: 0 0 255\n"
"colorentry 53: 0 0 255\n"
"colorentry 54: 0 0 255\n"
"colorentry 55: 128 128 128\n"
"colorentry 56: 128 128 255\n"
"colorentry 57: 0 128 0\n"
"colorentry 58: 0 0 128\n"
"colorentry 59: 0 0 255\n"
"colorentry 60: 128 0 128\n"
"colorentry 61: 0 128 0\n"
"colorentry 62: 0 128 0\n"
"colorentry 63: 0 128 64\n"
"colorentry 64: 0 0 128\n"
"colorentry 65: 0 0 128\n"
"colorentry 66: 255 0 255\n"
"colorentry 67: 128 128 0\n"
"colorentry 68: 0 0 128\n"
"colorentry 69: 0 0 255\n"
"colorentry 70: 0 0 128\n"
"colorentry 71: 0 0 255\n"
"colorentry 72: 0 0 0\n"
"colorentry 73: 255 255 255\n"
"colorentry 74: 192 192 192\n"
"colorentry 75: 0 255 255\n"
"colorentry 76: 0 0 0\n"
"colorentry 77: 128 0 0\n"
"colorentry 78: 128 128 128\n"
"colorentry 79: 128 128 0\n"
"colorentry 80: 255 0 255\n"
"colorentry 81: 0 0 0\n"
"colorentry 82: 0 0 255\n"
"colorentry 83: 0 0 0\n";

BOOL WINAPI DllEntryPoint(HINSTANCE hi,DWORD reason,LPVOID reserved) {
  if (reason==DLL_PROCESS_ATTACH)
    hinst=hi;
  return 1;
};

extc int _export cdecl ODBG_Plugindata(char shortname[32]) {
  strcpy(shortname,"OllyGraph");
  return PLUGIN_VERSION;
};

extc int _export cdecl ODBG_Plugininit( int ollydbgversion,HWND hw,ulong *features) {
  if (ollydbgversion<PLUGIN_VERSION)
    return -1;
  hwmain=hw;
  Addtolist(0,0,"OllyGraph plugin v0.1");
  Addtolist(0,-1,"  Copyright (C) 2004 Joe Stewart");
  return 0;
};

extc int _export cdecl ODBG_Pluginmenu(int origin,char data[4096],void *item) {
  switch (origin) {
    case PM_MAIN:
      strcpy(data,"0 Generate Function Flowchart,"
                  "1 Generate Function Call Graph|"
		  "2 Configure Path to Wingraph32,"
                  "3 About");
      return 1;
    case PM_DISASM:
      strcpy(data,"0 Generate Function Flowchart");
      return 1;
    default: break;
  };
  return 0;
};

extc void _export cdecl ODBG_Pluginaction(int origin,int action,void *item) {
  char filebuf[MAX_PATH];
  if (origin == PM_MAIN) {
    switch (action) {
      case 0:
	GenerateFunctionFlowchart();
        break;
      case 1:
	GenerateFunctionCallGraph();
        break;
      case 2:
        strcpy(filebuf,"wingraph32.exe");
        Browsefilename("Enter Path to wingraph32.exe",filebuf,".exe",0);
        Pluginwritestringtoini(hinst,"wingraph_path",filebuf);
        MessageBox(hwmain,
          "Path saved to .ini file","Path Saved",MB_OK|MB_ICONINFORMATION); 	
        break;
      case 3:
        MessageBox(hwmain,
          "OllyGraph plugin v0.1\n"
          "Generate VCG-compatible function graphs\n"
          "Copyright (C) 2004 Joe Stewart",
          "OllyGraph Plugin",MB_OK|MB_ICONINFORMATION);
        break;
      default: break;
    }; // end of switch
  } 
  if (origin == PM_DISASM){
    switch (action) {
      case 0:
	GenerateFunctionFlowchart();
	break;
      default: break;
    };
  }
}

void GenerateFunctionFlowchart(void) {
  HANDLE hTempFile; 
  DWORD dwBufSize=4096;
  char szTempName[MAX_PATH];
  char lpPathBuffer[4096];
  ulong start,end,psize,nextaddr,current,blocksize,currentnode,dsize;
  int writelen,orphannode;
  DWORD lenWritten;
  t_dump *cpuasm;
  char *decode;
  char *edgelist;
  char cmdbuf[MAXCMDSIZE];
  char szBuffer[TEXTLEN*2+4];
  char clean_comment[TEXTLEN*2];
  t_disasm_list_element *disasm_list_element;
  t_disasm_list_element *last_list_element = NULL; 
  void *disasm_list = NULL;
  void *next_disasm;
  t_disasm disasm_result;
  previous_edge_struct previous_edge;
  char wingraph_path[MAX_PATH];

  cpuasm = (t_dump *) Plugingetvalue(VAL_CPUDASM);
  start = 0; end = 0; nodes = 0;
  previous_edge.source = 0;
  previous_edge.targettrue = 0;
  previous_edge.targetfalse = 0;

  if (Getproclimits(cpuasm->sel0, &start, &end) == -1) {
   MessageBox(hwmain,
     "Address is not within known function boundaries\n"
     "(Did you run Analyze Code?)",
     "Function boundaries not found", MB_OK|MB_ICONEXCLAMATION);
    return;
  }

  edgelist = malloc(MAX_EDGELIST_SIZE);
    
  if (edgelist == NULL) {
     MessageBox(hwmain,
     "Could not allocate memory\n",
     "Out of Memory", MB_OK|MB_ICONEXCLAMATION);
     return;
  }

  memset(edgelist,0,MAX_EDGELIST_SIZE);

  GetTempPath(dwBufSize, lpPathBuffer); 
  cleanup_tempfiles(lpPathBuffer);
  GetTempFileName(lpPathBuffer,"ogh",0,szTempName);
  hTempFile = CreateFile((LPTSTR) szTempName, GENERIC_READ | GENERIC_WRITE, 
			  0,NULL, CREATE_ALWAYS,FILE_ATTRIBUTE_NORMAL,NULL); 

  if (hTempFile == INVALID_HANDLE_VALUE) { 
    MessageBox(hwmain, "Failed to create temporary file!", 
		       "Create Temp File Failed",MB_OK|MB_ICONINFORMATION);
    return;					         
  }

  currentnode = start;
  psize = 0;
  current = start; 
  blocksize = end - start;
  orphannode = 1;

  writelen = sprintf(szBuffer, "graph: {\ntitle: \"Graph of %x\"\n", start); 
  WriteFile(hTempFile, szBuffer, (DWORD) writelen, &lenWritten, NULL);
  WriteFile(hTempFile, vcg_params, (DWORD) strlen(vcg_params), &lenWritten, NULL);
  writelen = sprintf(szBuffer, "node: { title: \"%x\" vertical_order: 0 label: \"\f15%x:\f31", currentnode, currentnode); 
  WriteFile(hTempFile, szBuffer, (DWORD) writelen, &lenWritten, NULL);
  
  // first pass - disassemble instructions and enumerate nodes
  do {
    current += psize;
    decode = Finddecode(current,&dsize);
    
    Readcommand(current,cmdbuf);
    nextaddr = Disassembleforward(NULL,start,blocksize,current,1,1);
    psize = nextaddr - current;
    if (psize <= 0) {
	    psize = 1;
    } 

    disasm_list_element = malloc(sizeof(t_disasm_list_element));
     
    if (disasm_list_element == NULL) {
       MessageBox(hwmain,
	      "Could not allocate memory\n",
	      "Out of Memory", MB_OK|MB_ICONEXCLAMATION);
       free_list(disasm_list);
       return;
    }
    memset(disasm_list_element,0,sizeof(t_disasm_list_element));

    if (last_list_element != NULL) {
      last_list_element->next_element = disasm_list_element; 
    }
    last_list_element = disasm_list_element;

    // initialize pointer to first element
    if (disasm_list == NULL) { 
      disasm_list = disasm_list_element;
    }

    Disasm(cmdbuf,psize,current,decode,&(disasm_list_element->disasm),DISASM_ALL,NULL);
    disasm_result = disasm_list_element->disasm;
    // enumerate nodes
    if (currentnode == 0) {
      currentnode = current;
      append_nodelist(currentnode);
    }
    if ((disasm_result.jmpconst >= start) && (disasm_result.jmpconst < end)) {
      // this is a jump - start of an edge and pointer to a node
      append_nodelist(disasm_result.jmpconst);
      currentnode = 0; // update currentnode on next pass
    }
  } while (current + psize < end);

  // walk through saved disasm list and split into nodes
  next_disasm = disasm_list;
  currentnode = start;
  
  while (next_disasm != NULL) {
    disasm_list_element = next_disasm;
    next_disasm = disasm_list_element->next_element;
    disasm_result = disasm_list_element->disasm;
    current = disasm_result.ip;
    if (nodestart(current)) {
      if (orphannode == 1) {
        writelen = sprintf(szBuffer, "edge: { sourcename: \"%x\" targetname: \"%x\" }\n", currentnode, current);

        if (strlen(edgelist) + writelen >= MAX_EDGELIST_SIZE) {
          MessageBox(hwmain,
               "Maximum edges per function exceeded\n",
	       "Too many edges in graph", MB_OK|MB_ICONEXCLAMATION);
          free_list(disasm_list);
          return;
        }	   
        strcat(edgelist, szBuffer);
      }
      orphannode = 1;
      currentnode = current;
      writelen = sprintf(szBuffer, "\" }\n"); 
      WriteFile(hTempFile, szBuffer, (DWORD) writelen, &lenWritten, NULL);
      writelen = sprintf(szBuffer, "node: { title: \"%x\" label: \"\f15%x:\f31", current, current); 
      WriteFile(hTempFile, szBuffer, (DWORD) writelen, &lenWritten, NULL);
    }

    if (strlen(disasm_result.comment) > 0) {
      sanitize(disasm_result.comment, clean_comment);
      writelen = sprintf(szBuffer, "\n%s\t; %s", disasm_result.result, clean_comment); 
    } else {
      writelen = sprintf(szBuffer, "\n%s", disasm_result.result); 
    }
	
    WriteFile(hTempFile, szBuffer, (DWORD) writelen, &lenWritten, NULL);
    
    if (previous_edge.source != 0) {
      // append stored edge info with currentnode info to edgelist buffer
      previous_edge.targetfalse = currentnode;
      writelen = sprintf(szBuffer, "edge: { sourcename: \"%x\" targetname: \"%x\" label: \"false\" color: red }\n", previous_edge.source, previous_edge.targetfalse);
      if (strlen(edgelist) + writelen >= MAX_EDGELIST_SIZE) {
          MessageBox(hwmain,
          "Maximum edges per function exceeded\n",
          "Too many edges in graph", MB_OK|MB_ICONEXCLAMATION);
        free_list(disasm_list);
        return;
      }	   
      strcat(edgelist, szBuffer);
      writelen = sprintf(szBuffer, "edge: { sourcename: \"%x\" targetname: \"%x\" label: \"true\" color: darkgreen }\n", previous_edge.source, previous_edge.targettrue);
      if (strlen(edgelist) + writelen >= MAX_EDGELIST_SIZE) {
        MessageBox(hwmain,
          "Maximum edges per function exceeded\n",
          "Too many edges in graph", MB_OK|MB_ICONEXCLAMATION);
        free_list(disasm_list);
        return;
      }	   
      strcat(edgelist, szBuffer);
      previous_edge.source = 0;
    }

    disasm_result = disasm_list_element->disasm;

    if ((disasm_result.jmpconst >= start) && (disasm_result.jmpconst < end)) {
      // this is a jump - start of an edge and pointer to a node
      orphannode = 0;
      previous_edge.source = currentnode;
      previous_edge.targettrue = disasm_result.jmpconst;
      previous_edge.targetfalse = 0;
      if ((*disasm_result.result == 'J') && (*(disasm_result.result+1) == 'M')) {
        // straight jmp, no true/false
        writelen = sprintf(szBuffer, "edge: { sourcename: \"%x\" targetname: \"%x\" }\n", previous_edge.source, previous_edge.targettrue);

        if (strlen(edgelist) + writelen >= MAX_EDGELIST_SIZE) {
          MessageBox(hwmain,
               "Maximum edges per function exceeded\n",
	       "Too many edges in graph", MB_OK|MB_ICONEXCLAMATION);
          free_list(disasm_list);
          return;
        }	   
        strcat(edgelist, szBuffer);
	previous_edge.source = 0;
      }
    }
  }

  // close last node
  writelen = sprintf(szBuffer, "\" vertical_order: %d }\n",nodes); 
  WriteFile(hTempFile, szBuffer, (DWORD) writelen, &lenWritten, NULL);

  // write edgelist 
  WriteFile(hTempFile, edgelist, (DWORD) strlen(edgelist), &lenWritten, NULL);
  
  writelen = sprintf(szBuffer, "}\n"); 
  WriteFile(hTempFile, szBuffer, (DWORD) writelen, &lenWritten, NULL);

  CloseHandle(hTempFile);
  free_list(disasm_list);
  Pluginreadstringfromini(hinst,"wingraph_path",wingraph_path,"c:\\ida45\\wingraph32.exe");
  if ((int)ShellExecute(hwmain, "open", wingraph_path, szTempName, NULL, SW_SHOWNORMAL) < 32) {
   Error("Error while executing %s\n(check wingraph32 path configuration)", wingraph_path);   
  }  
}

void append_nodelist(ulong addr) {
  if (nodes > MAX_NODES) {
    return;
  }
  if (nodestart(addr)) {
    return;
  }
  nodelist[nodes] = addr;
  nodes++;
}

int nodestart(ulong addr) {
  int i;
  for (i = 0; i < nodes; i++) {
    if (nodelist[i] == addr) {
      return 1;
    }
  }
  return 0;
}

void sanitize(char *comment, char *cleaned) {
  char *t;

  for (t = cleaned; *comment; comment++) {
    if (*comment == '"') {
      *t++ = '\\';
      *t = '"';
    } else if (*comment == '\\') {
      *t++ = '\\';
      *t = '\\';
    } else {
      *t = *comment;
    }
    t++;
  }
  *t = '\0';
}  

void GenerateFunctionCallGraph(void) {
}

void GenerateXRefsToAddressGraph(void) {
  // not implemented
}

void GenerateXRefsFromAddressGraph(void){
  // not implemented
}

static void free_list(void *t_disasm_list) {
  void *next;
  t_disasm_list_element *tmp;
  
  if (t_disasm_list == NULL) {
    return;
  }
  tmp = t_disasm_list; 
  while (tmp->next_element != NULL) {
    next = tmp->next_element;
    free(tmp);
    tmp = next;
    if (tmp == NULL) {
      return;
    }
  }
}

void cleanup_tempfiles(char *tmppath)
{
   WIN32_FIND_DATA FindFileData;
   HANDLE hFind;
   char DirSpec[MAX_PATH];
   char FoundFile[MAX_PATH];
   DWORD dwError;

   strcpy (DirSpec, tmppath);
   strncat (DirSpec, "ogh*.tmp", 8);
   hFind = FindFirstFile(DirSpec, &FindFileData);

   if (hFind == INVALID_HANDLE_VALUE) 
   {
      return;
   } 
   else 
   {
      strcpy(FoundFile,tmppath);
      strcat(FoundFile,FindFileData.cFileName);
      DeleteFile(FoundFile);
      while (FindNextFile(hFind, &FindFileData) != 0) 
      {
	 strcpy(FoundFile,tmppath);
	 strcat(FoundFile,FindFileData.cFileName);
         DeleteFile(FoundFile);
      }
    
      dwError = GetLastError();
      if (dwError == ERROR_NO_MORE_FILES) 
      {
         FindClose(hFind);
      } 
   }
}

extc void _export cdecl ODBG_Pluginreset(void) {
};

extc int _export cdecl ODBG_Pluginclose(void) {
  return 0;
};

extc void _export cdecl ODBG_Plugindestroy(void) {
};
