#ifdef PROLOG

#define  NULL       0L   /* NULL pointer in compact and large */
#define  STR_LIMIT 120   /* Larger than any max string str_max */
#define  COL_MAX    77   /* Maximum length of line displayed */
#define  ROW_MAX    20   /* Maximum number of rows */
#define  OFFSET     20   /* Increment for offset display of long line */
#define  PAGE_SIZE  16   /* Size of page for page up/down */
#define  HALF (ROW_MAX/2)/* Row for middle line in split screen */
#define  LINK_MAX   32   /* Maximum number of hyper links per file */
#define  LINK_SIZE  40   /* Size of a hyper link field */
#define  LINK_CHAR  24   /* Up arrow displayed at hyper link field */
#define  HFB        11   /* Hyper field begin */
#define  HFE        14   /* Hyper field end */
#define  HDE        15   /* Hyper definition end */

typedef enum  {FALSE, TRUE} boolean;
typedef enum  {up, down, back, forwrd} direction;
typedef enum  {ch, wrd,  lin,  scrn, pg, txt} token;
typedef enum  {select_a, yes_a, no_a, rest_a, quit_a, esc_a} answers;

/*
   Structures:
      str_rec - string plus length
      line    - pointer to string plus doubly-link-list pointers
      arec    - activation record for stack of editor calls
*/

typedef struct {
       int           len;
       unsigned char s[STR_LIMIT];
     }  str_rec;
typedef str_rec *str_ptr;

static char buffer[LINK_SIZE+7]; /* For I/O of hyper fields */

static  int       normal_attr;   /* Attribute of nornal display */
static  int       reverse_attr;  /* Attribute for marking */
static  int       link_attr;     /* Link attribute */
static  boolean   local_rtl;     /* Language */

static boolean rtl(void) {return local_rtl;}
                                 /* Standalone version uses function */

extern void get_answer(int *);   /* Routines supplied by EDT.PRO */
extern void get_line(char **);
extern void demo_limit(void);
extern confirm_remove(int *);

/*
   Prolog library definitions.
   See spreds.h from Prof. User's Guide.
*/
#define field_attr SCR_FieldAttr_i
extern void field_attr(int, int, int, int);
#define scr_char SCR_ScrChar_i
extern void scr_char(int, int, char);
#define scr_char_1 SCR_ScrChar_o
extern void scr_char_1(int, int, char*);
#define str_len_2 PSTR_str_l_io
extern void str_len_2(char*, int*);
#define scr_attr SCR_ScrAttr_i
extern void scr_attr(int, int, int);
#define field_str SCR_FieldString_i
extern void field_str(int, int, int, char *);
#define cursor SCR_Cursor_i
extern void cursor(int row, int col);
#define movmem MEM_MovMem
extern void movmem(void *source, void *dest, unsigned noofbytes);
#define setmem MEM_SetMem
extern void MEM_SetMem(void *dest, unsigned len, unsigned char value);
#define beep OS_Beep
extern void beep(void);
#define scroll SCR_scroll
extern void scroll(int r, int c);
#define malloc MEM_AllocHeap
extern void *pascal MEM_AllocHeap(unsigned size);
#define free_mem MEM_ReleaseHeap
extern void pascal MEM_ReleaseHeap(void *ptr, unsigned size);
extern void *MEM_AllocGStack(unsigned size);
extern void *MEM_MarkGStack(void);
extern void MEM_ReleaseGStack(void *p);
extern void IO_WrStr(char *);

void mid_line(int i, boolean b) {}

#else

#include <alloc.h>
#include <stdlib.h>
#include <string.h>
#include <mem.h>
#include <conio.h>
#include <stdio.h>
#include "global.h"

#define DEFAULT_EXT    ".TEX"
#define DEFAULT_FILE   "ERROR.TEX"
#define VIEW_EXTENSION ".LIS"
#define BAK_EXTENSION  ".BAK"
#define NO_MEMORY      "No more memory"

static  FILE *fd;                 /* Text file for input/output */
static  char file_name[40];       /* File name saved for output */
static  char insert_name[40];     /* File name of inserted file */
static  char view_name[40];       /* File name for view of formatted */
static  char bak_name[40];        /* File name for back up copy */
static  boolean   view_flag;      /* Viewing formatted text */

static void cursor(int r, int c) {gotoxy(c+1, r+1);}
static void scroll(int i, int j) {if (i==1) scroll_up(); else scroll_down();}
static void beep(void) {bell();}
static void free_mem(void *p, int q) {free(p);}
static void demo_limit(void) {}
static void confirm_remove(int* i) {*i = confirm();}
extern void mid_line(int, boolean);

#endif

typedef enum  {not_inserting, begin_insert, end_insert} ins_type;
   /* For insertion of external file */
typedef enum  {cut, del_line, other_del} del_type;
   /* To decide which undelete to do */

typedef struct line_tag {
       struct   line_tag *up, *down;
       str_ptr  l;
     } line;
typedef line *line_ptr;

typedef struct arec_tag{
       struct    arec_tag *next;
       line_ptr  current;
       line_ptr  top;
       line_ptr  bottom;
       int       line_number;
       int       row;
       int       position;
       int       modified;
       int       edit_mode;
       char      links[LINK_MAX][LINK_SIZE];
     } arec;

/*
    Global variables
*/

static  line_ptr  top;          /* Top sentinel dummy node */
static  line_ptr  bottom;       /* Bottom sentinel dummy node */
static  line_ptr  current;      /* Current node */
static  line_ptr  my_free;      /* Pointer to top of free list */
static  arec      *save;        /* Pointer to list of activation records */
static  arec      *base;        /* Base activation record - for split */
static  arec      *active;      /* Active activation record */

static  line_ptr  begin_line;   /* Begin node of marked block */
static  int       begin_pos;    /* Position within begin node */
static  line_ptr  end_line;     /* End node of marked block */
static  int       end_pos;      /* Position within end node */
static  line_ptr  begin_paste;  /* Begin node of paste buffer */
static  line_ptr  end_paste;    /* End node of paste buffer */

static  int       current_line; /* Current line number */
static  int       mark_line;    /* Line number of beginning of marked block*/
static  int       row;          /* Current row */
static  int       position;     /* Current position within string */
static  int       row_first;    /* First row of display */
static  int       row_last;     /* Last row of display */
static  int       page_size;    /* Size of page limited to size of screen */
static  boolean   split;        /* Split screen or not */
static  boolean   join_cursor;  /* Joined cursor or not */

static  str_rec   deleted;      /* Store deleted string for undelte */
static  del_type  delete_flag;  /* Delete type for undelete */
static  boolean   mark_flag;    /* Are we marking? */
static  boolean   first_file;   /* First file - no editor data to save */
static  ins_type  insert_flag;  /* Current mode during insert external file*/

static  boolean   sensitive;    /* Case sensitive search and replace */
static  int       modified;     /* Modified flag to return to Prolog */
static  boolean   external_file;/* External file may delete links */
static  int       edit_mode;    /* Edit mode - read only or read/write */
static  int       str_max;      /* Maximum length of string */
static  int       offset;       /* Offset for display long line */
static  char      links[LINK_MAX][LINK_SIZE];  
				/* Store hyper fields */
static  int       demo;         /* Limit number of lines for demo */

void demo_init(int dem) {demo = dem;}

/******* Data structure manipulation *******/

static line_ptr alloc_line(void)
{
    line_ptr p;
  p         = malloc(sizeof(line));  /* Allocate the node */
  p->l      = malloc(str_max+3);     /* Allocate the string */
  p->l->len = 0;
  return p;
}

static void insert_line(void)
{
     line_ptr p;
        /* If free list empty, allocate from system */
  if (my_free == NULL) p = alloc_line();
  else {
    p = my_free;                 /* Use first free node */
    my_free = my_free->down;     /* Update free list pointer */
    p->l->len = 0;
  }
  p->up = current;               /* Doubly link new node after current */
  p->down = current->down;
  current->down->up = p;         /* Node after current to point up to p */
  current->down     = p;         /* Update current to point down to p */
}

static void delete_line(void)
{
     line_ptr after;                 /* Pointer to node after current */
  after = current->down;
  current->up->down = current->down; /* Doubly unlink current line */
  current->down->up = current->up;
  current->down     = my_free;       /* Link current to free list */
  my_free = current;                 /*   and update free list pointer */
  current = after;                   /* Now current is what was next line */
  position = 0;
}

static void move_line(int delta)
{
     int i;
  if (delta >= 0) {
    for (i = 1; i <=delta; i++)
      if (current != bottom) {    /* Check there is someplace to go */
        current = current->down;  /* Down for positive delta */
        current_line++;
      }
  }
  else
   for (i = -1; i >= delta; i--)
      if (current->up != top) {   /* Check there is someplace to go */
        current = current->up;    /* Up for negative delta */
        current_line--;
      }
  if (position > current->l->len) /* Do not position beyond end of string */
    position = current->l->len;
}

static void split_line(void)
{
     line_ptr  p;
     int n;
  insert_line(); /* Create the new line */
  p = current->down;
  if (position != current->l->len) {
    n = current->l->len - position;  /* Number of characters to move */
    movmem(&current->l->s[position], &p->l->s[0], n);
    current->l->len = position;      /* Update old line length */
    p->l->len = n;                   /* Set new line length */
  }
}

static void join_line(boolean *ok)
{
     line_ptr  p;
     int       n;
     int current_length;
  if (current == bottom->up) *ok = FALSE;
  else {
    p = current->down;                /* Next line */
    current_length = current->l->len; /* Save current length */
    n = current_length + p->l->len;   /* Joint length */
    if (n > str_max) *ok = FALSE;
    else {
      movmem(&p->l->s[0], &current->l->s[current_length], p->l->len);
      current->l->len = n;            /* Update current length */
      move_line(1);                   /* Move down to delete unneeded line */
      delete_line();
      move_line(-1);
      position = current_length;      /* Current position is old end of line */
      *ok = TRUE;
    }
  }
}

static void cut_lines(void)
{
     boolean ok;                /* Variable for calling join_line */
  current     = begin_line;     /* Set current location to marker and split*/
  position    = begin_pos;
  split_line();
  begin_paste = current->down;  /* Paste buffer will start after current */
  if (begin_line == end_line) { /* Cut within one line */
    end_line = current->down;   /* end of buffer same as begin */
    end_pos  = end_pos - current->l->len;
  }                             /* Subtract what was left in current */

  current    = end_line;        /* Set current location to marker and split*/
  position   = end_pos;
  split_line();
  end_paste  = current;         /* After split, current is last line cut */
  end_paste->down->up   = begin_paste->up;
                                /* Update doubly linked list */
  begin_paste->up->down = end_paste->down;
  current    = begin_paste->up; /* Update current location */
  position   = current->l->len;
  end_paste->down       = NULL; /* Used when pasting buffer */
  join_line(&ok);               /* Join remains of marked block begin, end */
}

/****** Display functions ******/

static void cursor_move(int r, int c)
/*
  Move cursor to row r, column c.
*/
{
  int col = c-offset;         /* Column relative to offset */
  int start = 0;              /* Language-dependent start column */
  int end = COL_MAX;          /* Language-dependent end column */

  if (rtl()) {
    col   = COL_MAX - col;
    start = COL_MAX - start;
    end   = COL_MAX - end;
  }

  if ((c >= offset) && (c < COL_MAX+offset)) cursor(r, col);
  else if (c < offset) cursor(r, start);
  else if (c >= COL_MAX+offset) cursor(r, end);
}

#ifdef PROLOG

void erase_string(int c, int nc)
/*
  Erase nc characters, starting from column c.
*/
{
  field_str(row, c, nc+1, " ");
  field_attr(row, c, nc+1, normal_attr);
}

static void write_string(char *p, int from, int to, int reverse)
/*
   Write string p, in columns from:to, possibly with reverse color.
*/
{
     int i, c, len, attr;
  if (reverse) attr = reverse_attr; else attr = normal_attr;
  if (to > from) {
  len = to - from;
  if (len+from-offset > COL_MAX)     /* Truncate length, if necessary */
    len = COL_MAX - from + offset;
  for (i = 0; i < len; i++) {
    if (rtl())                       /* Language-dependent column to write */
      c = COL_MAX - from + offset - i; 
    else 
      c = i + from - offset;
    if ((p[from+i] < LINK_MAX) && (external_file == FALSE)) {
      scr_char(row, c, LINK_CHAR);   /* If non-printing, must be link: */
      scr_attr(row, c, link_attr);   /* print arrow, unless external file */
    }
    else {
      scr_char(row, c, p[from+i]);   /* Normal character */
      scr_attr(row, c, attr);
    }
  }
  }
}

static void refresh(int, int);  /* Forward */

void change_language(void)
{
  local_rtl = ! local_rtl;
  refresh(row, position);
}

#else

void erase_string(int c, int nc)
{
     char blanks[COL_MAX];

    set_colors(fore_color, back_color);
    setmem(blanks, COL_MAX, ' ');
    blanks[nc] = '\0';
    gotoxy(c+1, row+1);
    cputs(blanks);
    putch(' ');
}

static void write_string(char *p, int from, int to, int reverse)
{
     int i, c, len;
  if (reverse) set_colors(back_color, fore_color);
  else set_colors(fore_color, back_color);
  if (to > from) {
  len = to - from;
  if (len+from-offset > COL_MAX) len = COL_MAX - from + offset;
  for (i = 0; i < len; i++) {
    if (rtl()) c = COL_MAX - from + offset - i; else c = i + from - offset;
    gotoxy(c+1, row+1);
    putch(p[from+i]);
  }
  }
}

static void refresh(int, int);  /* Forward */

void refresh_screen(void) {
  if (split) {
    other_window();
    refresh(row, position);
    other_window();
    mid_line(HALF+1, TRUE);
  }
  refresh(row, position);
}

#endif

static int after_mark(void)
/*
    Compute cursor location relative to (+1, -1, 0) mark location.
*/
{
  if (current_line > mark_line) return 1;
  else if (current_line < mark_line) return -1;
  else return (position - begin_pos);
}

static boolean check_mark_flag(void)
/*
    If marking, cursor move (etc.) will need to update display,
      but only if actually displaying and not past end of screen.
*/
{
  return (mark_flag && (position-offset <= COL_MAX+1));
}

static void write_current_line(int column, int line, int pos)
/*
     Write line pointed to by current starting from column.
     Very complicated due to update of marking attribute.
     The cursor may be elsewhere (see write_screen), so its
       position is given in line for marking updates.
*/
{
     boolean rev;
     int c;
  c = column - offset;
  if (c > COL_MAX) return;
  if (rtl()) erase_string(0, COL_MAX-c); else erase_string(c, COL_MAX-c);
  if (check_mark_flag()) {
    if (current_line == mark_line) {
      rev = after_mark() > 0;
      write_string((char *)current->l->s, column, begin_pos, rev);
      write_string((char *)current->l->s, begin_pos, current->l->len, !rev);
    }
    else if (current_line == line) {
      rev = after_mark() > 0;
      write_string((char *)current->l->s, column, pos, rev);
      write_string((char *)current->l->s, pos, current->l->len, !rev);
    }
    else {
      rev = ((current_line > line)  && (current_line < mark_line)) ||
        ((current_line < line)  && (current_line > mark_line));
      write_string((char *)current->l->s, column, current->l->len, rev);
    }
  }
    else write_string((char *)current->l->s, column, current->l->len, FALSE);
}

static void write_screen_1(int cursor_line, int save_pos)
/*
   Write screen starting at row, taking lines from current_line.
   Cursor_line used for computing reverse attributes if marking.
   Reset current_line after writing.
*/
{
    int save_row;
  save_row = row;
  do {      /* Until no more lines, or no more rows */
    write_current_line(offset, cursor_line, save_pos);
    move_line(1);
    row++;
  } while ((current != bottom) && (row <= row_last));
  while (row <= row_last) {erase_string(0, COL_MAX); row++;}
  row = save_row;
  position = save_pos;
  move_line(cursor_line - current_line);
  cursor_move(row, position);
}

static void write_screen(void) {
  write_screen_1(current_line, position);
}

static void refresh(int save_row, int save_position)
{
    int r = save_row - row_first;   /* Relative to split-screen */
  move_line(- r);
  row = row_first;
  write_screen();
  move_line(r);
  position = save_position;
  row = save_row;
  cursor_move(row, position);
}

void shift_right(void)
{
  offset -= OFFSET;
  if (offset < 0) offset = 0;
  mark_flag = FALSE;
  refresh(row, position);
}

void shift_left(void)
{
    int offmax = str_max/2;
  offset += OFFSET;
  if (offset > offmax) offset = offmax;
  mark_flag = FALSE;
  refresh(row, position);
}

static void my_scroll(int direction)
/*
  Hardware scroll if no split-screen, otherwise, re-write screen.
  direction: -1 = scroll down, 1 = scroll up.
*/
{
    int save_position;
    int save_row;
    int save_line;
  if (!split) {
    cursor_move(row_first, offset);
    scroll(direction, 0); 
    write_current_line(offset, current_line, position);
  }
  else {
    save_position = position;
    save_line = current_line; /* For reverse if marking */
    if (direction == 1) {
      save_row = row_last;    /* After scroll up, return to last row */
      move_line(- (row_last-row_first));
                              /* Move current line back to display rows */
    } else {
      save_row = row_first;   /* After scroll down, return to first row */
    }
    row = row_first;
    write_screen_1(save_line, save_position);
    position = save_position;
    row = save_row;
    cursor_move(row, position);
  }
}

/******** Cursor movement including goto line and mouse sync *******/

static void cursor_up(void)
{
     int n;
     boolean reverse;
  if (current != top->down) {
    if (check_mark_flag()) {
      n = begin_pos;
      if ((current_line == mark_line) & (n < position))
        write_string((char *)current->l->s, n, position, FALSE);
      else n = position;
      write_string((char *)current->l->s, offset,
        n, (current_line <= mark_line));
    }
    move_line(-1);
    if (row == row_first)
      my_scroll(-1);
    else {
      row--;
      if (check_mark_flag()) {
        n = begin_pos;
        if ((current_line == mark_line) && (n > position)) {
          write_string((char *)current->l->s, position, n, TRUE);
          reverse = FALSE;
        }
        else {
          n = position;
          reverse = current_line < mark_line;
        }
        write_string((char *)current->l->s, n, current->l->len, reverse);
      }
    }
    cursor_move(row, position);
  }
}

static void cursor_down(void)
{
     int n;
     boolean reverse;
  if (current != bottom->up) {
    if (check_mark_flag()) {
      n = begin_pos;
      if ((current_line == mark_line) && (n > position))
        write_string((char *)current->l->s, position, n, FALSE);
      else n = position;
      write_string((char *)current->l->s, n,
        current->l->len, (current_line >= mark_line));
    }
    move_line(1);
    if (row == row_last) {
      my_scroll(1);
    }
    else {
      row++;
      if (check_mark_flag()) {
        n = begin_pos;
        if ((current_line == mark_line) && (n < position)) {
        write_string((char *)current->l->s, offset, n, FALSE);
          reverse = TRUE;
        } else {
          n = offset;
          reverse = (current_line > mark_line);
        }
        write_string((char *)current->l->s, n, position, reverse);
      }
    }
    cursor_move(row, position);
  }
}

static void cursor_home(void);   /* Forward declaration */

static void cursor_forward(void)
{
  if (position != current->l->len) {
    position++;
    if (check_mark_flag())
      write_string((char *)current->l->s, position-1,
      position, (after_mark() > 0) );
    cursor_move(row, position);
  }
  else if (current != bottom->up) {
    cursor_down();
    if (position != 0) cursor_home();
  }
}

static void cursor_end(void)
{
  if ((position == current->l->len) && (current != bottom->up))
    cursor_down();
  while (position != current->l->len) cursor_forward();
}

static void cursor_back(void)
{
  if (position != 0) {
    position--;
    cursor_move(row, position);
    if (check_mark_flag())
      write_string((char *)current->l->s, position,
        position+1, (after_mark() < 0));
    cursor_move(row, position);
  }
  else if (current != top->down) {
    cursor_up();
    if (position != current->l->len) cursor_end();
  }
}

static void cursor_home(void)
{
  if ((position == 0) && (current != top->down)) cursor_up();
  while (position != 0) cursor_back();
}

static int count_forward(boolean include_blanks)
/*
   Return number of characters until next word (or eol), 
     skipping leading blanks.
*/
{
     int n;
  n = position;
  while ((current->l->s[n] != ' ')  && (n < current->l->len)) n++;
  if (include_blanks)
    while ((current->l->s[n] ==  ' ') && (n < current->l->len)) n++;
  return (n - position);
}

static int count_back(void)
/*
    Starting before current position (e.g. beginning of word),
      skip back over blanks, then non-blanks.
    Do not count beyond beginning of line, or initial blanks.
*/
{
     int n;
  n = position - 1; 
  while ((current->l->s[n] == ' ')  && (n > 0)) n--;
  while ((current->l->s[n] != ' ')  && (n > 0)) n--;
  if ((n < 0) || (current->l->s[n] == ' ')) n++;
  return (position - n);
}

static void cursor_to_line(int l)
/*
   Move current_line to line l (offset by page_size) and display.
*/
{
  move_line(l - current_line);
  move_line(-page_size);
  row = row_first;
  write_screen();
}

void cursor_to_line_number(int);   /* Forward */

void move_cursor(token t, direction d)
{
    int r, j, n, i;
  switch (t) {
    case ch:
      switch (d) {
        case up:     cursor_up();     break;
        case down:   cursor_down();   break;
        case back:   cursor_back();   break;
        case forwrd: cursor_forward(); break;
      }
      break;
    case wrd:
      switch (d) {
        case forwrd:
           n = count_forward(TRUE);
           if (n <= 0) n = 1;
           for (i = 1; i <= n; i++) cursor_forward();
           break;
        case back:
           n = count_back();
           if (n <= 0) n = 1;
           for (i = 1; i <= n; i++) cursor_back();
           break;
      }
      break;
    case lin:
      switch (d) {
        case forwrd: cursor_end(); break;
        case back:   cursor_home(); break;
      }
      break;
    case scrn:
      switch (d) {
        case up:
          if (position != 0) cursor_home();
          r = row - row_first;
          for (j = 1; j <= r; j++) cursor_home();
          break;
        case down:
          if (position != current->l->len) cursor_end();
          r = row_last - row;
          for (j = 1; j <= r; j++) cursor_end();
          break;
      }
      break;
    case pg:
      i = position;
      switch (d) {
    case up:    cursor_to_line_number(current_line - page_size); break;
    case down:  cursor_to_line_number(current_line + page_size); break;
      }
      position = i;
      if (position > current->l->len) position = current->l->len;
      cursor_move(row, position);
      break;
    case txt:
      switch (d) {
        case up:    cursor_to_line(1); cursor_home(); break;
        case down:  cursor_to_line(30000); move_cursor(scrn, down); break;
      }
      break;
  }
}

void cursor_to_line_number(int l)
/*
   Move cursor to line number, set cursor at page or at end.
*/
{
     int i, n;
  if (demo != 0) if (l > demo) {demo_limit(); return;}
  cursor_to_line(l);
  if (l > page_size) n = page_size; else n = l - 1;
  for (i = 1; i <= n; i++) move_cursor(ch, down);
}

void mouse_sync(int r, int c, int *ok)
/*
  Move cursor to (r, c) mouse co-ordinates.
  ok if in range.
*/
{
     int i;
  r--; c--;
  if ((r >= row_first) && (r <= row_last) && (c >= 0) && (c <= COL_MAX)) {
    *ok = 1;
    if (r < row)
      for (i = row; i > r; i--) cursor_up();
    else
      for (i = row; i < r; i++) cursor_down();
    if (rtl()) c = COL_MAX - c;
    if (c > current->l->len) c = current->l->len;
    if (c < position)
      for (i = position; i > c; i--) cursor_back();
    else
      for (i = position; i < c; i++) cursor_forward();
  }
  else *ok = 0;
}

/******* Insertion, new line, deletion, undelete *******/

void insert_char(unsigned char c, int ltr)
/*
    Insert character c at current position.
    If ltr, do not move cursor.
    If no more room, split line before inserting.
    Split location:
      Try to move back to beginning of word,
        else try to move forward to end of word (or end of line).
      If nothing works, just split here.
*/
{
     int i, n;
     direction move;
  if (mark_flag || !edit_mode) return;
  modified = TRUE;
  if (current->l->len == str_max) {
    move = up;  /* dummy */
    n = count_back();
    if (n != position) move = back;
    else {
      n = count_forward(TRUE);
      if (n == 0) move = down;
      else if (n != current->l->len - position) move = forwrd;
    }
    if      (move == forwrd) for (i = 1; i <= n; i++) cursor_forward();
    else if (move == back)   for (i = 1; i <= n; i++) cursor_back();
    split_line();
    if (offset > 0) {
      offset = 0;
      refresh(row, position);
    }
    write_screen();
    if      (move == forwrd) for (i = 1; i <= n;   i++) cursor_back();
    else if (move == back)   for (i = 1; i <= n+1; i++) cursor_forward();
    else if (move == down)   cursor_forward();
  }
  movmem(&current->l->s[position], &current->l->s[position+1],
         current->l->len - position);
  current->l->s[position] = c;
  current->l->len++;
  write_current_line(position, current_line, position);
  cursor_move(row, position);
  if (ltr == 0)  cursor_forward();
}

void new_line(void)
{
  if (mark_flag || !edit_mode) return;
  if (demo != 0) if (current_line > demo) {demo_limit(); return;}
  modified = TRUE;
  split_line();
  write_screen();
  cursor_down();
  position = 0;
  cursor_move(row, position);
}

static void remove_link(void);  /* Forward */

static void delete_forward(int n)
{
     int i;
  if (n > 1)  {
    if (external_file == FALSE)
      for (i = 0; i < n; i++)    /* Do not delete string containing link */
        if (current->l->s[position+i] < LINK_MAX) {beep(); return;}
    movmem(&current->l->s[position],  &deleted.s[0], n);
    deleted.len = n;
    delete_flag = other_del;
  } else {
    if ((external_file == FALSE) && (current->l->s[position] < LINK_MAX)) {
      confirm_remove(&i);
      if (i != 0) remove_link();
      return;
    }
  }
  movmem(&current->l->s[position+n], &current->l->s[position],
         current->l->len-(position+n));
  current->l->len -= n;
  write_current_line(offset, current_line, position);
  cursor_move(row, position);
}

void delete_for_index(boolean *OK) {
  int i, n;
  if (mark_flag) {
    if (begin_line != current) {*OK = FALSE; return;}
    if (begin_pos > position) {
      i = begin_pos; begin_pos = position; position = i;}
    while ( (current->l->s[position-1] == ' ') && (position > 0) ) 
      cursor_back();
    n = position - begin_pos;
    for (i = 1; i <= n; i++) cursor_back();
  } else n = count_forward(FALSE); 
  if (n <= 0) {*OK = FALSE; return;}
  delete_forward(n);
  *OK = TRUE;
}

void delete (token t, direction d)
{
      int i;                /* Index for cursor movement */
      int n;                /* Count to delete */
      boolean ok;           /* Success flag for join_line */
  if (mark_flag || !edit_mode) return;
  modified = TRUE;
  switch (d) {
    case forwrd:
      if (position == current->l->len) {
        join_line(&ok);
        if (ok) write_screen();
        else beep();
      }
      else {
        switch(t) {
          case ch:   n = 1; break;
          case wrd:  n = count_forward(TRUE); if (n <= 0) n = 1; break;
          case lin:  n = current->l->len - position; break;
        }
        delete_forward(n);
      }
      break;

    case back:
      if (position == 0)
        if (current == top->down) beep();
        else { cursor_back(); delete(t, forwrd); }
      else {
        switch (t) {
          case ch:   n = 1; break;
          case wrd:  n = count_back(); 	if (n <= 0) n = 1; break;
          case lin:  n = position; break;
        }
        for (i = 1; i <= n; i++) cursor_back();
        delete_forward(n);
      }
      break;

    case up:                           /* Used for delete entire line */
      if (external_file == FALSE)
        for (i = 0; i < current->l->len; i++)
      if (current->l->s[i] < LINK_MAX) {beep(); return;}
      if (position != 0) cursor_home();
      if (position != current->l->len) delete(lin, forwrd);
      if (current == bottom->up) delete(ch, back);
      else delete(ch, forwrd);
      delete_flag = del_line;
      break;
  }
}

static void undelete(void)
{
     int i;
  if (delete_flag == del_line) split_line();
  for (i = 1; i <= deleted.len; i++)
    insert_char(deleted.s[i-1], FALSE);
  if (delete_flag == del_line) {
    write_screen();
    move_cursor(ch, forwrd);
  }
}

void reverse_lines(void)
/*
  Reverse all lines in edit buffer.
*/
{
    line_ptr p;
    char *s;
    int len, i;
    char temp;
  p = top->down;
  while (p != bottom) {
    s = p->l->s;
    len = p->l->len;
    for (i = 0; i < len/2; i++) {
      temp = s[len-1-i];
      s[len-1-i] = s[i];
      s[i] = temp;
    }
    p = p->down;
  }
  move_cursor(txt, up);
  modified = TRUE;
}

/********** Search and replace ********/

char to_upper(char c) {
/*
    If not case-sensitive search, translate to upper-case before searching.
*/
  if ((!sensitive) && (c >= 'a') && (c <= 'z')) return (c - 32);
  else return c;
}

void compare_string( 
      char *s1,         /* Search for string s1 */
      char *s2,         /*   within s2 */
      int len,          /* Length of s2 */
      boolean *fnd,     /* return found or not */
      int *index)       /* and the index within s2 where found */
/*
  Search for first letter of s1 in s2,
    then check if s1 could fit within s2,
    then check all of s1 against s2.
*/
{
     int i;             /* Local index within s2 */
     int k;             /* Index within s1 */
     boolean found;     /* Local found flag */
  i = 0;
  found = FALSE;
  while ((!found) && (i < len)) {
    while ((s1[0] != to_upper(s2[i])) && (i < len)) i++;
    if (i == len) break;
    if (s1[1] == 0) {found = TRUE; break;}
    k = 1;
    while ((s1[k] == to_upper(s2[i+k])) && (s1[k] != 0)) k++;
    found = (s1[k] == 0);
    if (!found) i++;
  }
  *fnd = found;
  *index = i;
}

static void find_string(
     str_rec str,      /* String to search for */
     int from,         /* starting position in current line */
     boolean *fnd,     /* found or not */
     int *line,        /* line number where found */
     int *pos)         /* position in line where found */
{
   boolean found;      /* Local found flag */
   boolean last_line;  /* Is this the last line? */
   int first_char;     /* Local first character position */
   int line_count;     /* Line counter during search */
   int index;          /* Local index where found */
   line_ptr next;      /* Loop variable for lines */

  first_char = from;
  line_count = 0;
  found = FALSE;
  next = current;
  last_line = FALSE;
  str.s[str.len] = 0;
  while ((!found) && (!last_line)) {
    compare_string(str.s, &next->l->s[first_char],
      next->l->len-first_char, &found, &index);
    index += first_char;
    first_char = 0;
    if (!found) {
      if (next->down == bottom) last_line = TRUE;
      else {
       last_line = FALSE;
       next  = next->down;
      }
      line_count++;
    }
  }
  *fnd = found;
  *line = line_count;
  *pos = index;
}

static void find(
   str_rec str,         /* String to search for */
   int     again,       /* Used if search again command is entered */
   boolean *fnd)        /* Found or not */
{
    boolean found;      /* Found or not */
    int i;              /* Counter for lines to skip */
    int skip;           /* Number of lines to skip until found */
    int pos;            /* Position within found line */
    direction d;        /* Skip direction relative to position in line */

  find_string(str, position+again, &found, &skip, &pos);
  if (found) {
    if ((skip > 0) && (skip + row <= row_last))
      for (i = 1; i <= skip; i++) move_cursor(ch, down);
    else
      cursor_to_line_number(current_line + skip);
    skip = pos - position;
    if (skip < 0) {d = back; skip = - skip;} else d = forwrd;
    for (i = 1; i <= skip; i++) move_cursor(ch, d);
  }
  *fnd = found;
}

void search_string(char *s, int l, int again, boolean *fnd)
{
     boolean found;
     str_rec str;
  movmem(s, str.s, l);
  str.len = l;
  find(str, again, &found);
  if (!found) cursor_move(row, position);
  *fnd = found;
}

static void replace(str_rec str1, str_rec str2)
{
    boolean found;            /* String found flag */
    answers a     = no_a;     /* Answer to user prompt */
    int     again;            /* Search again or for first occurence */
    str_rec remain;           /* Holds remainder of string while replacing */

  find(str1, 0, &found);
  while (found && (a < quit_a)) {
    if (a != rest_a) {
      write_screen();
      get_answer((int *) &a);
      if (a == no_a) again = 1; else again = str2.len;
    }
    if (((a == rest_a) || (a == yes_a))  &&
	((current->l->len + str2.len - str1.len) < str_max) ) {
      remain.len = current->l->len - position - str1.len;
      movmem(&current->l->s[position+str1.len], &remain.s[0], remain.len);
      movmem(&str2.s[0], &current->l->s[position], str2.len);
      movmem(&remain.s[0], &current->l->s[position+str2.len], remain.len);
      current->l->len = position + str2.len + remain.len;
      write_current_line(0, current_line, position);
    }
    find(str1, again, &found);
  }
}

void replace_string(char *s1, int l1, char *s2, int l2)
{
    str_rec str1, str2;
    int save_row, save_position, save_line;
  if (mark_flag || !edit_mode) return;
  modified = TRUE;
  save_position = position;
  save_row = row;
  save_line = current_line;
  movmem(s1, str1.s, l1);
  str1.len = l1;
  movmem(s2, str2.s, l2);
  str2.len = l2;
  replace(str1, str2);
  move_line(save_line - current_line);
  current_line = save_line;
  refresh(save_row, save_position);
}

/******** Global variable initialization, save, restore *********/

static void init_variables(void)
{
  top            = alloc_line();
  bottom         = alloc_line();
  top->down      = bottom;
  bottom->up     = top;
  top->up        = NULL;
  bottom->down   = NULL;
  current        = top;
  position       = 0;
  current_line   = 0;
  modified       = FALSE;
  setmem(links, LINK_SIZE*LINK_MAX, ' ');
}

static void update_struc(arec* p)
/*
  Store current values into activation record p.
*/
{
  p->position = position;
  p->row = row;
  p->current = current;
  p->top = top;
  p->bottom = bottom;
  p->line_number = current_line;
  p->modified = modified;
  p->edit_mode = edit_mode;
}

static void save_struc(void)
/*
   Create a new activation record, store values and link in chain.
*/
{
     arec *p;
  p = malloc(sizeof(arec));
  update_struc(p);
  movmem(links, p->links, LINK_SIZE*LINK_MAX);
  p->next = save;
  if (save == NULL) base = p;
  save = p;
  active = p;
}

static void restore_struc(arec *q, boolean fresh)
/*
   Restore values from activation record q.
   if fresh, refresh the screen.
*/
{
  active = q;
  modified = q->modified;
  edit_mode = q->edit_mode;
  current = q->current;
  top = q->top;
  bottom = q->bottom;
  current_line = q->line_number;
  movmem(q->links, links, LINK_SIZE*LINK_MAX);
  row = q->row;
  position = q->position;
  if (row < row_first) row = row_first;
  else if (row > row_last) row = row_last;
  if (fresh) refresh(row, position);
}

static void restore_mem(void)
/*
  Link edit buffer into free list.
*/
{
  bottom->down = my_free;
  my_free = top;
}

static void restore(boolean fresh)
/*
   Restore latest activation record:
     Restore memory, restore values and free activation record.
*/
{
    arec* q = save;
  restore_mem();
  restore_struc(q->next, fresh);
  save = q->next;
  free_mem(q, sizeof(arec));
}

void get_split(int *s) {
/*
   Interface procedure to Prolog programs.
*/
  *s = split;
}

void split_screen(boolean fresh)
/*
  Toggle split screen.
  If split screen, restore latest activation record on full screen.
  If not, restore first record on upper window
    and latest record on lower window.
  fresh: refresh screen or not.
*/
{
  if (split) {
    row_first = 0;
    row_last  = ROW_MAX;
    page_size = PAGE_SIZE;
    split     = FALSE;
    mid_line(HALF+1, FALSE);
    save->row = 0;
    restore_struc(save, fresh);
  }
  else if (save != base) {
    row_first = 0;
    row_last  = HALF - 1;
    page_size = PAGE_SIZE / 2;
    restore_struc(base, TRUE);
    mid_line(HALF+1, TRUE);
    row_first = HALF + 1;
    row_last  = ROW_MAX;
    restore_struc(save, fresh);
    split = TRUE;
    join_cursor = FALSE;
  }
}

void other_window(void)
/*
  Go to other window by saving current values
    and restoring from activation record (first or latest).
*/
{
  if (split) {
    if (row_first == 0) {
      update_struc(base);
      row_first = HALF + 1;
      row_last  = ROW_MAX;
      restore_struc(save, TRUE);
      active = save;
    } else {
      update_struc(save);
      row_first = 0;
      row_last  = HALF - 1;
      restore_struc(base, TRUE);
      active = base;
    }
  } else {
    if (active == base) {
      update_struc(base);
      restore_struc(save, TRUE);
    } else {
      update_struc(save);
      restore_struc(base, TRUE);
    }
  }
}

void goto_other_window(void) {
/*
    If you are in the upper window and there is a lower window, go there.
*/
  if (split && (row_first == 0)) other_window();
}  

void split_cursor(token t, direction d)
/*
   Cursor move: if join_cursor, move cursors in both windows.
*/
{
  if (join_cursor && split) {
    other_window();
    move_cursor(t, d);
    other_window();
    move_cursor(t, d);
  }
  else move_cursor(t, d);
}

void toggle_joint_cursor(void) {
  join_cursor = ! join_cursor;
}

/********** Mark, cut, copy, paste, insert ********/

void cancel_command(void)
/*
  Called when ESC is pressed during marking or inserting.
*/
{
  if (insert_flag != not_inserting) {
    mark_flag = FALSE;
    insert_flag = not_inserting;
    if (split) split_screen(TRUE);
    restore(TRUE);
  }
  else if (mark_flag) {
    mark_flag = FALSE;
    refresh(row, position);
  }
}

void mark_block(void)
{
  mark_flag = TRUE;
  begin_line = current;
  begin_pos  = position;
  mark_line = current_line;
}

static void insert_block(void)
/*
   Insert block at current location:
     Split current line
     Loop through paste buffer, 
       insert fresh line after current and move text from paste buffer
     Join to suffix and prefix of old current line
*/
{
     line_ptr p;              /*Iterates over inserted lines*/
     line_ptr save_current;   /*Current is changed for insert_line call*/
     boolean  ok;             /*Result of join_line*/
  save_current = current;
  split_line();
  p = begin_paste;
  while (p != NULL) {
    insert_line();
    current = current->down;
    movmem(&p->l->s[0], &current->l->s[0], p->l->len);
    current->l->len = p->l->len;
    p = p->down;
  }
  position = current->l->len;
  join_line(&ok);
  current = save_current;
  position = current->l->len;
  join_line(&ok);
}

static void reverse_markers(
   line_ptr *p1, line_ptr *p2,
   int *n1, int *n2)
/*
  Reverse markers:
    Normally, you begin to mark a block and then move the cursor down.
    If you moved the cursor up, this void exchanges the
      begin and end node pointers (p1, p2) and positions (n1, n2)
      so that begin is before end.
    Since pointers are used comparison cannot be done,
      instead, search down from begin and if you reach bottom
      before end, the exchange must be done.
*/
{
     int i;
     line_ptr  q;
  if (*p1 == *p2) {              /* Begin and end in same line */
    if (*n1 > *n2)  {i = *n1; *n1 = *n2; *n2 = i;}
  }
  else {
    q = *p1;
    while ((q != *p2) && (q != bottom)) q = q->down;
    if (q == bottom) {           /* Hit bottom before end */
      q = *p1; *p1 = *p2; *p2 = q;
      i = *n1; *n1 = *n2; *n2 = i;
      mark_line = current_line;  /* Also, update line number */
    }
  }
}

static boolean check_block_memory(unsigned long int mem,
      line_ptr begin, line_ptr end) {
   line_ptr p = begin;
   int      lines = 0;
   int      increment = sizeof(line) + str_max + 3;
  while (p != end) {
    lines++;
    p = p->down;
  }
  p = my_free;
  while ((p != NULL) && (lines > 0)) {
    lines--;
    p = p->down;
  }
  if (lines == 0) return TRUE;
  else if ((mem/increment) < (unsigned long int) lines) {
      beep();
      return FALSE;
  }
  else return TRUE;
}

void cut_copy_block(int copy, unsigned long int mem)
/*
     Cut is copy = 0.
  Cut the lines beginning with the marked block begin and end.
  Since the marking may begin or end in the middle of a line,
    split the first and last line (if necessary).
  Set paste buffer pointers to point to cut lines and
    then relink what remains.
  Join the split lines.
  Decrement the number of cut lines from the total line count.
  If copy, (re-)insert block just after cutting.
*/
{
     int delta_r;
/*  if (!mark_flag) return; */
  if (!mark_flag && (copy != 0)) {
    move_cursor(txt, up);
    mark_block();
    move_cursor(txt, down);
    cut_copy_block(1, mem);
    return;
  }
  if (!edit_mode && (copy == 0)) return;
  if (begin_line == NULL) return;
  end_line = current;
  end_pos  = position;
  reverse_markers(&begin_line, &end_line, &begin_pos, &end_pos);
  if (begin_paste != NULL) {
    end_paste->down = my_free;
    my_free = begin_paste;
    begin_paste = NULL;
  }
  if ((copy == 1) &&
      (!check_block_memory(mem, begin_line, end_line))) return;
  mark_flag = FALSE;
  cut_lines();
  delete_flag = cut;
  if (copy == 0) modified = TRUE; else insert_block();
  if (begin_pos < 0)  position = 0;
  else if (begin_pos > str_max)  position = str_max;
  else position = begin_pos;
  delta_r = current_line - mark_line;
  if (row  - delta_r < row_first)  row = row_first;
  else row -= delta_r;
  current_line = mark_line;
  write_screen();
}

void paste_block(unsigned long int mem)
{
  if (!edit_mode) return;
  if (demo != 0) if (current_line > demo) {demo_limit(); return;}
  if (delete_flag != cut) {undelete(); return;}
  if (begin_paste == NULL) return;
  if (!check_block_memory(mem, begin_paste, end_paste)) return;
  insert_block();
  modified = TRUE;
  write_screen();
}

#ifdef PROLOG

/******** Hyper field processing ********/

static void enter_link(char *l, unsigned char *number)
/*
    Store link string l in link database entry number.
    Return LINK_MAX if no more room.
*/
{
     int i;
  i = 0;
  while ((i < LINK_MAX) && (links[i][0] != ' ')) i++;
  if (i < LINK_MAX) {
    movmem(l, links[i], LINK_SIZE);
    }
  *number = (unsigned char) i;
}

void define_link(char *l, int *ok)
/*
    Store link string l in link database entry number.
    Return ok = 1 (successful), = 0 (no more room)
*/
{
     unsigned char number;
  if (!edit_mode) return;
  enter_link(l, &number);
  if (number < LINK_MAX) {
    insert_char(number, 0);
    *ok = 1;
    }
  else *ok = 0;
}

void active_link(char **s, int *ok)
/*
    Return pointer s to active link string
    Return ok = 1 (successful), = 0 (no such link)
*/
{
     unsigned char c;
  c = current->l->s[position];
  if (c >= LINK_MAX) *ok = 0;
  else {
    movmem(links[(int) c], &buffer[6], LINK_SIZE);
    *s = &buffer[3];
    *ok = 1;
  }
}

void get_one_link(int i, char *s)
/*
  Return in s link number i.
*/
{
    movmem(links[i], s, LINK_SIZE);
}

static void remove_link(void)
/*
    Remove link at current position, if exists.
*/
{
     char c;
  if (!edit_mode) return;
  c = current->l->s[position];
  if (c < LINK_MAX) {
    links[c][0] = ' ';
    current->l->s[position] = ' ';
    delete(ch, forwrd);
  }
}

void strip_hyper_fields(void)
/*
   Strip hyper field data from paste buffer.
*/
{
     line_ptr q;
     str_ptr  p;
     int      i;
  q = begin_paste;
  while (q != NULL) {
    p = q->l;
    i = 0;
    while (i < p->len) {
      if (p->s[i] < LINK_MAX)
	movmem(&p->s[i+1], &p->s[i], (p->len--)-i);
      i++;
    }
    q = q->down;
  }
}

/****** Input, output *****/

static void read_file(char *s, boolean ext_file)
/*
      Text to edit is in s.
      If ext_file, repeatedly call get_line to get next line in s.
      Hyperfield format: HFB...text...HFEAAA...link...HDE,
        where AAA is the display attribute.
*/
{
   str_ptr p;                      /* Pointer to internal string */
   int     i;                      /* Index into line string */
   int     c;                      /* Character buffer */
   int     n;                      /* Input string index */
    
   unsigned char *hyper_loc;       /* Pointer to location to insert link */
   boolean hyper_def;              /* Are we in hyper def field? */
   int     hyper_index;            /* Index within hyper field buffer */
   char    hyper_field[LINK_SIZE]; /* Hyper field buffer */

  hyper_def = FALSE;
  n = 0;
  if (ext_file == TRUE) get_line(&s);
  c = s[n++];
  while (c != 0) {
    insert_line();
    move_line(1);
    p = current->l;
    i = 0;
    while ((c != '\n') && (c != 0)) {
      if (i == str_max-1) {  /* Split line that is too long */
        p->s[i++] = c;
        p->len = i;
        insert_line();
        move_line(1);
        p = current->l;
        i = 0;
      } else {
        p->s[i] = c;
        if (c == HFB)
          hyper_loc = &p->s[i++];  /* Save hyperfield begin location */
        else if (c == HFE) {
          hyper_def = TRUE;
          hyper_index = 0;
          n += 3;                  /* Skip over attribute field */
          c = s[n];
        }
        else if (hyper_def && (c == HDE)) {
          hyper_def = FALSE;       /* End of hyperfield, enter link */
          enter_link(hyper_field, hyper_loc);
        }
        else if (hyper_def && (hyper_index < LINK_SIZE))
          hyper_field[hyper_index++] = c;  /* Save link field */
        	else i++;
      }
      c = s[n++];
    }
    p->len = i;
    if (ext_file == TRUE) {get_line(&s); n = 0;}
    if (c != 0) c = s[n++];
  }
  if (p->len != 0) insert_line();
  current        = top;
  position       = 0;
  current_line   = 0;
}

void my_ed_text(char **s, int ext_file)
/*
     Write text to string s (or to file if ext_file = 0).
*/
{
    line_ptr q;        /* Line index */
    str_ptr  p;        /* Pointer to internal string */
    int      i;        /* Index of chars within line */
    int      n;        /* Output string index */
    char     *a;       /* Pointer to output string */
    unsigned char  c;  /* Character buffer */
    char     buf[200]; /* For external file */

  if (ext_file == 0) a = &buf[0];
  else {               /* Compute size of string to output: */
    n = 0;             /*   all lines plus max links */
    q = top->down;
    while (q != bottom) {
      n += q->l->len + 1;
      q = q->down;
    }
    for (i = 0; i < LINK_MAX; i++)
      if (links[i][0] != 0) n += (LINK_SIZE + 6);
    a = MEM_AllocGStack(n+1);
  }
  q = top->down;
  n = 0;
  while (q != bottom) {
    p = q->l;
    for (i = 0; i <= p->len-1; i++) {
      c = p->s[i];
      if (c < LINK_MAX) {  /* Link character, move link to output */
       movmem(links[(int) c], &buffer[6], LINK_SIZE);
       if ((i == p->len) || (p->s[i+1] < LINK_MAX)) buffer[1] = ' ';
       else {i++; buffer[1] = p->s[i];}
       movmem(buffer, &a[n], LINK_SIZE+7);
       n += LINK_SIZE+7;
      }
      else a[n++] = c;
    }
    a[n++] = '\n';
    if (ext_file == 0) {
      a[n] = '\0';
      IO_WrStr(a);
      n = 0;
    }
    q = q->down;
  }
  if (ext_file == 0) *s = buffer;  /* Arbitrary address, not used */
  else {a[n] = 0; *s = a;}
}

void insert_file(char *s)
/*
  Insert text from file s.
  Insert_flag implements a state machine:
    not, begin, end
*/
{
  if (!edit_mode) return;
  if (insert_flag == not_inserting) {
    update_struc(save);
    offset = 0;
    init_variables();
    read_file(s, 0);
    move_cursor(txt, up);
    save_struc();
    insert_flag = begin_insert;
  }
  else if (insert_flag == begin_insert) {
    mark_block();
    insert_flag = end_insert;
  }
  else if (insert_flag == end_insert) {
    cut_copy_block(FALSE, 0);
    if (split) split_screen(TRUE);
    restore(TRUE);
    insert_flag = not_inserting;
  }
}

/******* Main interface procedures **********/

void my_ed_open(char *s, int mode, int initial_rtl, int ext_file)
{
  if (save != NULL) update_struc(save);
  first_file = FALSE;
  init_variables();
  edit_mode = mode;
  local_rtl = initial_rtl;
  if (ext_file == 0) external_file = TRUE; else external_file = FALSE;
  offset = 0;
  read_file(s, external_file);
}

void my_ed_open_1(int pos, boolean fresh)
/*
     Set position to pos by counting line lengths until correct line found.
*/
{
     line_ptr p;
     int count;
  if (!fresh) save_struc();
  else {
    move_cursor(txt, up);
    save_struc();
    p = current;
    count = 0;
    while (pos >= p->l->len) {
      pos = pos - p->l->len - 1;
      p = p->down;
      count++;
    }
    if (count > 0) cursor_to_line_number(count+1);
    for (count = pos; count > 0; count--) move_cursor(ch, forwrd);
  }
}

/*
void paste_buffer_in(char *s) {
  init_variables();
  read_file(s, 0);
  if (begin_paste != NULL) {
    end_paste->down = my_free;
    my_free = begin_paste;
  }
  begin_line = top->down;
  begin_pos = 0;
  end_line = bottom->up;
  end_pos = end_line->l->len;
  cut_lines();
  delete_flag = cut;
  restore(TRUE);
}
*/

void paste_buffer_out(char **s) {
/*
    Copy paste buffer to string s.
*/
    line_ptr save_first;
  if (begin_paste == NULL) {
    *s = "";
    return;
  }
  save_first = top->down;
  top->down = begin_paste;
  end_paste->down = bottom;
  my_ed_text(s, 1);
  top->down = save_first;
  end_paste->down = NULL;
}

void my_ed_close(boolean fresh)
{
     line_ptr q, q1;
  strip_hyper_fields();
  if (save != base) {
    if (split && (save->next == base)) split_screen(fresh);
    restore(fresh);
  }
  else {
    free_mem(save, sizeof(arec));
    save = NULL;
    first_file = TRUE;
    bottom->down = my_free;
    my_free = NULL;
    q = top;
    while (q != NULL) {
      q1 = q->down;
      MEM_ReleaseHeap(q->l, str_max+3);
      MEM_ReleaseHeap(q,    sizeof(line));
      q = q1;
    }
  }
}

void my_ed_status(int *mark, int *mod, int *mode)
{
  *mark = mark_flag;
  *mod  = modified;
  *mode = edit_mode;
}

void my_ed_init(int case_sens, int s_max, int normal_a,
                int rev_a, int link_a, char *link_attr_string)
{
  sensitive      = case_sens;
  str_max        = s_max;
  normal_attr    = normal_a;
  reverse_attr   = rev_a;
  link_attr      = link_a;
  begin_paste    = NULL;
  end_paste      = NULL;
  begin_line     = NULL;
  end_line       = NULL;
  begin_pos      = 0;
  end_pos        = 0;
  my_free        = NULL;
  save           = NULL;
  base           = NULL;
  active         = base;
  first_file     = TRUE;
  row            = row_first;
  deleted.len    = 0;
  mark_flag = FALSE;
  insert_flag = not_inserting;
  local_rtl = FALSE;
  buffer[0] = HFB; buffer[2] = HFE;
  buffer[3] = link_attr_string[0];
  buffer[4] = link_attr_string[1];
  buffer[5] = link_attr_string[2];
  buffer[LINK_SIZE+6] = HDE;
  demo      = 0;
  row_first = 0;
  row_last  = ROW_MAX;
  page_size = PAGE_SIZE;
  split    = FALSE;
}

#else

void remove_link(void) {}

void read_file(boolean initial, char *s)
{
    str_ptr p;         /*Pointer to internal string*/
    char    name[40];  /*File name*/
    str_rec str;       /*File name returned by get_string*/
    boolean ok;        /*Is file opened?*/
    int     i;         /*Index into string*/
    int     c;         /*Character buffer*/
    char    *pt;

  if (!initial && (insert_flag == not_inserting)) {
    strcpy(view_name, file_name);
    pt = strchr(view_name, '.');
    memmove(pt, VIEW_EXTENSION, 5);
    view_flag = TRUE;
    strcpy(name, view_name);
  }
  else if (initial && (strlen(s) == 0)) {
    str.len = 0;
    get_string(file_p, &str);
    memmove(name, &str.s[0], str.len);
    name[str.len] = 0;
  }
  else strcpy(name, s);
  if (strchr(name, '.') == NULL)
    strcpy(name, strcat(name, DEFAULT_EXT));

  fd = fopen(name, "r");
  ok = (fd != NULL);
  if (ok) {
    c = fgetc(fd);
    while (c != EOF) {
      if (coreleft() < 1000L) {
        clear_screen(LIGHTGRAY, BLACK);
	cputs(NO_MEMORY);
        exit(1);
      }
      insert_line();                   /*Insert a new line*/
      move_line(1);                    /*Move current pointer down*/
      p = current->l;
      i = 0;
      while ((c != '\n') && (c != EOF) && (i < str_max)) {
        p->s[i] = c;
        i++;
        c = fgetc(fd);
      }
      p->len = i;
      if ((i != str_max) || ((i == str_max) && (c == '\n'))) c = fgetc(fd);
    }
    fclose(fd);
  }
 if (initial) strcpy(file_name, name); else strcpy(insert_name, name);
 if (!ok) insert_line();
}

boolean inserting(void) {
  return insert_flag != not_inserting;
}

void insert_file(void)
{
    str_rec str;

  if (edit_mode && (insert_flag == not_inserting)) {
    update_struc(save);
    insert_flag = begin_insert;
    init_variables();
    str.len = strlen(insert_name);
    memmove(&str.s[0], insert_name, str.len);
    get_string(file_p, &str);
    memmove(insert_name, &str.s[0], str.len);
    insert_name[str.len] = 0;
    read_file(FALSE, insert_name);
    move_cursor(txt, up);
    save_struc();
  }
}

void my_ed_open(char *s, int pos, int mode, int initial_rtl)
{
     line_ptr p;
     int count;
  if (save != NULL) update_struc(save);
  init_variables();
  set_rtl(initial_rtl);
  edit_mode = mode;
  read_file(first_file, s);
  first_file = FALSE;
  move_cursor(txt, up);
  save_struc();
  p = current;
  count = 0;
  while (pos >= p->l->len) {
    pos = pos - p->l->len - 1;
    p = p->down;
    count++;
  }
  if (count > 0) cursor_to_line_number(count+1);
  for (count = pos; count > 0; count--) move_cursor(ch, forwrd);
}

void write_file(void)
{
    str_ptr p;             /*Pointer to internal string*/
    int     i;             /*Index into string*/
    char    *pt;
    char    *f;

  if (active == base) f = file_name; else f = insert_name;
  current = top->down;
  position = 0;
  current_line = 1;
  if (view_flag) fd = fopen(view_name, "w");
  else {
    strcpy(bak_name, f);
    pt = strchr(bak_name, '.');
    memmove(pt, BAK_EXTENSION, 5);
    remove(bak_name);
    rename(f, bak_name);
    fd = fopen(f, "w");
  }
  if (fd == NULL) fd = fopen(DEFAULT_FILE, "w");
  if (fd != NULL) {
    while (current != bottom) {
      p = current->l;
      for (i = 0; i <= p->len-1; i++) fputc(p->s[i], fd);
      fputc('\n', fd);
      move_line(1);                    /*Down to next line*/
    }
    fclose(fd);
  }
}

void save_file(void)
{
   int save_line = current_line;
   int save_pos  = position;
  write_file();
  move_line(save_line - current_line);
  current_line = save_line;
  position = save_pos;
  refresh(row, position);
}

void my_ed_close(void)
{
     line_ptr q, q1;
  view_flag = FALSE;
  if (save != base) restore(TRUE);
  else {
    free_mem(save, sizeof(arec));
    save = NULL;
    first_file = TRUE;
    bottom->down = my_free;
    my_free = NULL;
    q = top;
    while (q != NULL) {
      q1 = q->down;
      free(q->l);
      free(q);
      q = q1;
    }
  }
}

void print_file(void)
{
      char pr[60];

    strcpy(pr, "PRINT ");
    strcat(pr, view_name);
    system(pr);
}

extern void format(char *);

void format_file(void)
{
  write_file();
  format(file_name);
  move_cursor(txt, up);
}

void my_ed_init()
{
  begin_paste    = NULL;
  end_paste      = NULL;
  begin_line     = NULL;
  end_line       = NULL;
  begin_pos      = 0;
  end_pos        = 0;
  my_free        = NULL;
  save           = NULL;
  base           = NULL;
  active         = base;
  first_file     = TRUE;
  view_flag      = FALSE;
  mark_flag      = FALSE;
  insert_flag    = not_inserting;
  row            = row_first;
  deleted.len    = 0;
  offset         = 0;
  set_rtl(FALSE);
  str_max        = 97;
  sensitive      = 0;
  demo           = 0;
  row_first      = 0;
  row_last       = ROW_MAX;
  page_size      = PAGE_SIZE;
  split          = FALSE;
  external_file  = TRUE;
  insert_name[0] = '\0';
}

#endif

