/*
    CFORM.C
      If the symbol PROLOG is defined, a PDC module is created,
      otherwise a standalone program for debugging.
*/

#define  L_L           'l'       /* left, right center for tabular */
#define  L_C           'c'
#define  L_R           'r'
#define  R_L           ''
#define  R_C           ''
#define  R_R           ''

#define  STR_MAX       150       /* Maximum size of formatted line */
#define  MAX_FOOT     1024       /* Maximum size of footnote */
#define  MAX_FOOT_NUM    6       /* Maximum size of footnote number */
#define  MIN_SPACE_ON_PAGE 8     /* Minimum space on page for new section */
#define  TABLE_MAX     12        /* Maximum columns per table entry */
#define  COLUMN_CHAR   '&'       /* Table column separator */
#define  COMMAND_CHAR  '\\'      /* LaTeX command character */
#define  LEFT_BRACE    '{'
#define  RIGHT_BRACE   '}'
#define  LF            '\n'
#define  ZERO          '\0'
#define  BLANK         ' '
#define  PERIOD        '.'
#define  FF            '\f'

#define  LINE_MIN     40
#define  LINE_MAX    120
#define  MAR_MIN       0
#define  MAR_MAX      25
#define  PAGE_MIN     30
#define  PAGE_MAX    100
#define  TCW_MIN       4
#define  TCW_MAX      20
#define  TOP_MIN       0
#define  TOP_MAX      15

typedef enum {FALSE, TRUE} boolean;
typedef enum {NO_IT, START_IT, DURING_IT, STOP_IT} italic_type;
    /* Italic processing is difficult because must turn on and off
          at each row so that margin is not underlined. */

typedef enum {
  center_b, center_e, quote_b, quote_e, table_b, table_e,
  rtl_c, nortl_c, newpage_c, parms_c,
  enum_b, enum_e, item_c, single_c, double_c,
  eol_c, italic_b, italic_e, foot_c, index_c} command_type;

typedef struct {
       command_type command_id;   /* Command type */
       boolean line_command;      /* Ignore rest of line after command? */
       int lang;                  /* 0 = english, 1 = hebrew */
       int len;                   /* Significant characters in command */
       char *str;                 /* The command string */
     } command_record;

static int   PAGE_SIZE;      /* Lines per page */
static int   LINE_SIZE;      /* Characters per line */
static int   MARGIN_SIZE;    /* Characters per margin */
static int   INDENT;         /* Extra space for indenting */
static int   TOP;            /* Space at top of page */
static char  IT_START[8];    /* Italic start sequence */
static char  IT_STOP[8];     /* Italic stop sequence */
static char  IT_START_REV[8];/* Italic start sequence */
static char  IT_STOP_REV[8]; /* Italic stop sequence */
static int   TABLE_WIDTH;    /* Width of each column in table */

static char  w[STR_MAX];     /* Write line buffer */
static int   w_len;          /* Write line length */
static char  *r;             /* Pointer to input buffer */
static int   n;              /* Current index in read line buffer */
static char  blanks[STR_MAX];/* Blanks ready-to-use */

static int   page;           /* Current page number */
static int   line;           /* Current line number */
static int   margin;         /* Current margin, changed by quote */
static int   width;          /* Current width, changed by quote */
static int   words;          /* Word counter for justify */
static int   foot_number;    /* Footnote number counter */
static int   additional;     /* Additional chars. = escapes, etc. */
static int   enum_number;    /* Enumeration paragraph counter */
static int   italic_max;     /* Maximum size of italic escape code */

static int   spacing;        /* Single or double space */
static int   columns;        /* Columns in table */
static char  column_type[TABLE_MAX];  /* Left, right or center */

static boolean   right;          /* Extra spaces from right */
static boolean   centering;      /* Flag for centering text */
static boolean   rtl;            /* Flag for left-to-right languages */
static boolean   enum_written;   /* Has enumeration number been written */
static boolean   enum_flag;      /* In enumeration environment */
static boolean   footnote_flag;  /* In footnote, ignore italic */
static boolean   table_flag;     /* Tabular environment */
static boolean   page_top;       /* Page number on top or not */
static boolean   justify_lines;  /* Justify lines or not */
static italic_type italic_flag;  /* State of italic processing */
static boolean   was_italic;     /* Was italic in this line */
static boolean   italic_input;   /* Difference between }} and italic_e */

static void str_int(char *, int);         /* Forward declarations */
static int process_command(int *, int);

#ifdef PROLOG
extern int search_command(char *);       /* From CPROC.C */
extern void get_command(int, command_record*);

/*
   PDC library interface
*/
extern void IO_WrCh(char);
extern void IO_WrStr(char *);
#define str_len STR_StrLen
extern int STR_StrLen(char*);
#define movmem MEM_MovMem
extern void MEM_MovMem(void *source, void *dest, unsigned noofbytes);
#define setmem MEM_SetMem
extern void MEM_SetMem(void *dest, unsigned len, unsigned char value);

extern void save_footnote(char *, int);    /* Defined in EDT.PRO */
extern void save_index(char *, int, int);

#else

#include "stdio.h"
#include "stdlib.h"
#include "string.h"
#include "mem.h"

/*
   LaTeX command strings
*/
#define  S_CENTER_B    "\\begin{center}"
#define  S_CENTER_E    "\\end{center}"
#define  S_QUOTE_B     "\\begin{quote}"
#define  S_QUOTE_E     "\\end{quote}"
#define  S_TABLE_B     "\\begin{tabular}{lrc}"
#define  S_TABLE_E     "\\end{tabular}"
#define  S_RTL         "\\hebrew"
#define  S_NORTL       "\\english"
#define  S_ITALIC_B    "{\\it{}}"
#define  S_ITALIC_E    "}}"
#define  S_FOOTNOTE    "\\footnote{}"
#define  S_INDEX       "\\index{}"
#define  S_NEWPAGE     "\\newpage"
#define  S_ENUM_B      "\\begin{enumerate}"
#define  S_ENUM_E      "\\end{enumerate}"
#define  S_ITEM        "\\item"
#define  S_SINGLE      "\\single"
#define  S_DOUBLE      "\\double"
#define  S_EOL         "\\\\"
#define  S_PARMS       "\\parameters{60,10,56,5,8,8}"

#define  R_CENTER_B    "\\}{"
#define  R_CENTER_E    "\\}{"
#define  R_QUOTE_B     "\\}{"
#define  R_QUOTE_E     "\\}{"
#define  R_TABLE_B     "\\}{}{"
#define  R_TABLE_E     "\\}{"
#define  R_RTL         "\\"
#define  R_NORTL       "\\"
#define  R_ITALIC_B    "}\\}{{"
#define  R_ITALIC_E    "{{"
#define  R_FOOTNOTE    "\\}{"
#define  R_INDEX       "\\}{"
#define  R_NEWPAGE     "\\"
#define  R_ENUM_B      "\\}{"
#define  R_ENUM_E      "\\}{"
#define  R_ITEM        "\\"
#define  R_SINGLE      "\\"
#define  R_DOUBLE      "\\"
#define  R_EOL         "\\\\"
#define  R_PARMS       "\\}06,01,65,5,8,8{"

static  command_record commands[] = {
   {italic_b,  FALSE, 0,  5, S_ITALIC_B},
   {italic_b,  FALSE, 1,  6, R_ITALIC_B},
   {italic_e,  FALSE, 0,  2, S_ITALIC_E},
   {italic_e,  FALSE, 1,  2, R_ITALIC_E},
   {foot_c,    FALSE, 0, 10, S_FOOTNOTE},
   {foot_c,    FALSE, 1,  6, R_FOOTNOTE},
   {index_c,   FALSE, 0,  7, S_INDEX},
   {index_c,   FALSE, 1,  6, R_INDEX},
   {item_c,    FALSE, 0,  5, S_ITEM},
   {item_c,    FALSE, 1,  5, R_ITEM},
   {eol_c,     FALSE, 0,  2, S_EOL},
   {eol_c,     FALSE, 1,  2, R_EOL},
   {center_b,  TRUE,  0, 14, S_CENTER_B},
   {center_b,  TRUE,  1, 11, R_CENTER_B},
   {center_e,  TRUE,  0, 12, S_CENTER_E},
   {center_e,  TRUE,  1, 10, R_CENTER_E},
   {quote_b,   TRUE,  0, 13, S_QUOTE_B},
   {quote_b,   TRUE,  1, 12, R_QUOTE_B},
   {quote_e,   TRUE,  0, 11, S_QUOTE_E},
   {quote_e,   TRUE,  1, 11, R_QUOTE_E},
   {table_b,   TRUE,  0, 15, S_TABLE_B},
   {table_b,   TRUE,  1, 11, R_TABLE_B},
   {table_e,   TRUE,  0, 13, S_TABLE_E},
   {table_e,   TRUE,  1, 10, R_TABLE_E},
   {rtl_c,     TRUE,  0,  7, S_RTL},
   {rtl_c,     TRUE,  1,  6, R_RTL},
   {nortl_c,   TRUE,  0,  8, S_NORTL},
   {nortl_c,   TRUE,  1,  7, R_NORTL},
   {newpage_c, TRUE,  0,  8, S_NEWPAGE},
   {newpage_c, TRUE,  1,  5, R_NEWPAGE},
   {enum_b,    TRUE,  0, 17, S_ENUM_B},
   {enum_b,    TRUE,  1, 12, R_ENUM_B},
   {enum_e,    TRUE,  0, 15, S_ENUM_E},
   {enum_e,    TRUE,  1, 11, R_ENUM_E},
   {single_c,  TRUE,  0,  7, S_SINGLE},
   {single_c,  TRUE,  1,  5, R_SINGLE},
   {double_c,  TRUE,  0,  7, S_DOUBLE},
   {double_c,  TRUE,  1,  5, R_DOUBLE},
   {parms_c,   TRUE,  0, 12, S_PARMS},
   {parms_c,   TRUE,  1,  9, R_PARMS},
   {-1,        FALSE, 0,  0, ""},        /* Sentinel for end */
   };

#define  BUFFER_SIZE   30000u
#define  DEFAULT_EXT   ".TEX"
#define  LIST_EXT      ".LIS"
#define  NO_FILE       "Format which file?"
#define  BAD_INPUT     "No such file"
#define  BAD_OUTPUT    "Cannot create output file"
#define  TOO_LONG      "Too long, truncating ..."

  unsigned char buf[BUFFER_SIZE];  /* Read file to this buffer */
  FILE      *infile;
  FILE      *outfile;

static void IO_WrCh(char c)      {fputc(c, outfile);}
static void IO_WrStr(char *s)    {fputs(s, outfile);}
static unsigned str_len(char *s) {return strlen(s);}
void frm_init(char *, char*, int, boolean, boolean);
void frm_section(char *, int);
#define str_len_2 PSTR_str_l_io
void str_len_2(char* c, int* i) {*c = *i;};
void *MEM_AllocGStack(unsigned s) {return (void *) s;};

static void save_footnote(char *foot, int l)
{
  foot[l] = LF;
  foot[l+1] = ZERO;
  fputs(foot, stdout);
}

static void save_index(char *index, int l, int pg)
{
    char buffer[5];
  str_int(buffer, pg);
  fputs(buffer, stdout);
  index[l] = LF;
  index[l+1] = ZERO;
  fputs(index, stdout);
}

int search_command(char *s)
/*
   Search for a command.
   If invalid, will terminate on zero-length command and return -1.
*/
{
      int i, cmd = -1;
      boolean ok;
   do {
     cmd++;
     ok = TRUE;
     for (i = 0; ok && (i < commands[cmd].len); i++)
       ok &= (s[i] == commands[cmd].str[i]);
   } while (!ok);
   return cmd;
}

void get_command(int i, command_record *c)
{
  *c = commands[i];
}

static void open_files(char *s)
{
     char *period;
     char file_name[40];
  strcpy(file_name, s);
  if (strchr(file_name, PERIOD) == NULL)
    strcpy(file_name, strcat(file_name, DEFAULT_EXT));
  infile = fopen(file_name, "r");
  if (infile == NULL) { puts(BAD_INPUT); exit(EXIT_FAILURE); }
  period = strchr(file_name, PERIOD);
  strcpy(period, LIST_EXT);
  outfile = fopen(file_name, "w");
  if (outfile == NULL) { puts(BAD_OUTPUT); exit(EXIT_FAILURE); }
}

void format(char *s)
{
     char a[STR_MAX-1];
     unsigned int i = 0;
     int l;
  open_files(s);
  while ((fgets(a, STR_MAX, infile) != NULL) && (i < BUFFER_SIZE)) {
    l = str_len(a);
    movmem(a, &buf[i], l);
    i += l;
  }
  fclose(infile);
  if (i >= BUFFER_SIZE) puts(TOO_LONG);
  frm_init("\x1b-1", "\x1b-0", 0, FALSE, TRUE);
  frm_section(buf, 1);
  fclose(outfile);
}

#endif

static void blank_line(char *s) {setmem(s, STR_MAX, BLANK);}

static void str_int(char *s, int i)
{
     char buffer[10];
     int  k, count = 0;
   do {
      buffer[count++] = '0' + i % 10;
      i = i / 10;
    } while (i != 0);
    for (k = 0; k < count; k++) s[k] = buffer[count - k - 1];
    s[k] = ZERO;
}

static void write_spaces(int n)
{
  if (n <= 0) return;
  blanks[n] = ZERO;
  IO_WrStr(blanks);
  blanks[n] = BLANK;
}

static void italic_write(char *s)
/*
    Called after writing margin and enum in english to start italic.
    Called after writing line in hebrew to stop italic before enum.
*/
{
  if ((italic_flag == DURING_IT) || (italic_flag == STOP_IT) ||
      (rtl && was_italic))
    IO_WrStr(s);
  if      (italic_flag == START_IT)  italic_flag = DURING_IT;
  else if (italic_flag == STOP_IT)   italic_flag = NO_IT;
  was_italic = FALSE;
}

static void write_out(void)
{
  #define E_MAX  3   /* Maximum size of enumeration number+1 */
  int i;
  char e_buf[E_MAX];

  if (rtl) {
    write_spaces(margin);
    if ((italic_flag == DURING_IT) || (italic_flag == START_IT) )
       IO_WrStr(IT_START);            /* After margin start italic */
    if (centering) write_spaces((width - w_len + additional) / 2);
    else write_spaces(width - w_len + additional - 1);
       /* Extra spaces before short line */
    for (i = w_len-1; i >= 0; i--) IO_WrCh(w[i]);
    italic_write(IT_STOP);            /* Now stop italic */
    if ( (enum_flag) && (!enum_written) ) {
      str_int(e_buf, enum_number);
      write_spaces(E_MAX - str_len(e_buf));
      IO_WrCh(PERIOD);
      IO_WrStr(e_buf);
      enum_written = TRUE;
    }
  }
  else {
    if ( (enum_flag) && (!enum_written) ) {
      str_int(e_buf, enum_number);
      write_spaces(margin - E_MAX - 1);
      IO_WrStr(e_buf);
      IO_WrCh(PERIOD);
      write_spaces(E_MAX - str_len(e_buf));
      enum_written = TRUE;
    }
    else write_spaces(margin);
    if (centering) write_spaces((width - w_len) / 2);
    italic_write(IT_START);        /* Start italic after margin and enum */
    w[w_len] = ZERO;
    IO_WrStr(w);
    if (italic_flag == DURING_IT)  /* Stop italic after intermediate lines */
      IO_WrStr(IT_STOP);
  }
}

static void write_page_number(void)
{
    char ww[STR_MAX];
    int i = MARGIN_SIZE + (LINE_SIZE - 4) / 2;

    blank_line(ww);
    ww[i] = '-';
    str_int(&ww[i+2], page);
    while (ww[i] != ZERO) i++;
    ww[i++] = BLANK;
    ww[i++] = '-';
    ww[i++] = ZERO;
    IO_WrStr(ww);
    IO_WrCh(LF);
}

void bottom_of_page(int end_page_flag)
{
  if (end_page_flag != 0) {
    if ((!page_top) && (end_page_flag >= 1)) {
      if (end_page_flag == 1)
        do {IO_WrCh(LF);} while (line++ <= PAGE_SIZE);
      IO_WrCh(LF);
      write_page_number();
    }
    IO_WrCh(FF);
    page++;
    line = 1;
  }
}

void print_top_of_page(void)
{
  int i;

  if (line != 1) return;
  for (i = 0; i < TOP; i++) IO_WrCh(LF);
  if (page_top && (page != 1)) {
    write_page_number();
    IO_WrCh(LF);
    IO_WrCh(LF);
  }
  right = TRUE;
}

void check_space_on_page(void)
/*
   Called from DOCU.PRO.
   If very few lines on page, do a page break and update Prolog values.
*/
{
  if ( (PAGE_SIZE - line) < MIN_SPACE_ON_PAGE) {
     bottom_of_page(TRUE);
     print_top_of_page();
  }
}

static void skip_blanks(int *i)
{
  while (r[*i] == BLANK) (*i)++;
}

static int line_length(void)
{
     int i = 0;
  while ( (r[n+i] != LF) && (r[n+i] != ZERO) ) i++;
  return i;
}

static void break_line(void)
/*
   Break current line unless widow:
     This is the last line on the page and the line is
       non-null but also not full.
*/
{
  if (w_len == 0) {
    IO_WrCh(LF);
    line++;
    if (line > PAGE_SIZE) {
      bottom_of_page(TRUE);
      print_top_of_page();
    }
  }
  else {
    if (w[w_len-1] == BLANK) w_len--;

    if ( ( (line > PAGE_SIZE) && (w_len >= width-1) ) ||  /* Widow */
      (line > PAGE_SIZE+1) ) {
      bottom_of_page(TRUE);
      print_top_of_page();
    }
    write_out();
    IO_WrCh(LF);
    line++;
    if (spacing == 2) {IO_WrCh(LF); line++;}
    w_len = 0;
    additional = 0;
    blank_line(w);
    words = 0;
  }
}

static void justify(void)
/*
  Justify the line in w:
    Compute array of extra spaces.
    Copy w temporarily to temp and then copy back,
      inserting spaces as required.
    Change direction of distribution of extra spaces each line,
      to avoid uneven white space.
*/
{
    int excess;                  /* Excess spaces to distribute */
    int evenly;                  /* Even distribution of spaces */
    int spaces[40];              /* Extra spaces after each word */
    char temp[STR_MAX];          /* Copy of w before inserting spaces */
    int k;                       /* Word index */
    int i = 0, j = 0;            /* Indices into temp and w */

  if (!justify_lines) return;
  words--;                       /* Intervals=number of words-1 */
  if (words == 0) return;        /* Something wrong, but don't bomb */
  excess = width - w_len + additional;  /* Excess spaces to distribute */
  evenly = excess / words;       /* Number that evenly distribute */
  for (k = 0; k <= words-1; k++) /* Even distribution */
    spaces[k] = 1 + evenly;      /* Don"t forget original space */

  excess = excess % words;       /* Excess left after even distribution */
  if (right)                     /* Depending on direction, */
    k = words - 1;               /*   distribute from last word, */
  else k = 0;                    /*   or from first word */
  while (excess > 0) {
    spaces[k]++;
    excess--;
    if (right) k--; else k++;
  }
  right = !right;                /* Change direction for next line */

  movmem(&w, &temp, w_len);      /* Copy w to temp. variable and blank */
  blank_line(w);
  for (k = 0; k <= words; k++) {
    while (temp[j] != BLANK) w[i++] = temp[j++];
				 /* Move word from temp to w */
    if (k != words)              /* Except for last word, */
      i = i + spaces[k];         /*   add blanks */
    j++;                         /* Skip over single blank in temp */
  }
  w_len = i;
}

static void find_break(int start, int *len, char *ch)
/*
   Find the next break in the input.
   Return the number of characters to the break and the break character.
      e.g. xyz\ gives len = 3, char = \
*/
{
     int l = 0;
     char c;
  while (TRUE) {
    c = r[start+l];
    if (c==LF || c==BLANK || c==ZERO ||
	c==COMMAND_CHAR || c==LEFT_BRACE || c==RIGHT_BRACE) break;
    else l++;
  }
  *len = l;
  *ch  = c;
}

static void find_end(int start, int lang, int *l)
/*
   Find the end of a command.
   According to language, look for start character { or }.
   Return the length to the end and store where indicated.
   Recurse on internal command.
*/
{
     int k;
     int s = start;
     char c = BLANK;
     char end_char;
     int  ok;
  if (lang == 0) end_char = RIGHT_BRACE; else end_char = LEFT_BRACE;
  while (c != end_char) {
    find_break(s, &k, &c);
    if ( (c==LEFT_BRACE) || (c==RIGHT_BRACE) || (c==COMMAND_CHAR) ) {
      ok = process_command(&s, k);
      if (ok >= 0) c = BLANK;   /* Force continue on loop */
    }
    else if ( (c==BLANK) || (c==LF) ) ok = -1;
    else {      /* ZERO, we have run off end of input */
      s += k;
      break;
    }
    if (ok < 0) s += k + 1;  /* Add up so far and continue */
  }
  *l = s - 1 - start;
}

static void get_parm(int *parm, int *start, int lang, int min, int max)
{
     int i;
     int j = 1;
     int k = 0;
     char c;
  do {
    c = r[(*start)++];
  } while ((c != LF) && ((c < '0') || (c > '9')));
  if (c != LF)
    do {
      i = c - '0';
      if (lang == 0) k = (k*10) + i;
      else {k = k + j*i; j *= 10;}
      c = r[(*start)++];
    } while ((c != LF) && (c >= '0') && (c <= '9'));
  if (k < min) k = min; else if (k > max) k = max;
  *parm = k;
}

static void store_word(int start, int l)
{
  if (w_len - additional + l >= width) {
    justify();
    break_line();
  }
  movmem(&r[start], &w[w_len], l);
  w_len += l;
}

static void end_of_word(int start)
/*
   For italic, footnote, index, end of command may be end of word:
      ... on the Nile\index{Nile} where ...
*/
{
  if ( (r[start] == BLANK) || (r[start] == LF) ) {
    w[w_len++] = BLANK;
    words++;
  }
}

static void create_foot_number(char *s, int *len)
/*
   From footnote number num, create a string s of len.
   Parentheses according to language.
*/
{
     char buffer[MAX_FOOT_NUM];
     int i, j;
  footnote_flag = TRUE;
  str_int(buffer, ++foot_number);
  j = str_len(buffer);
  if (rtl) {
    s[0] = ')';
    for (i = 0; i < j; i++) s[i+1] = buffer[j - i - 1];
    s[j+1] = '(';
  }
  else {
    s[0] = '(';
    for (i = 0; i < j; i++) s[i+1] = buffer[i];
    s[j+1] = ')';
  }
  s[j+2] = BLANK;   /* Trailing blank after number */
  *len = j+2;
}

static void italic_command(command_type id)
/*
   Insert italic start or stop, according to language.
   Additional characters must not be considered when 
     breaking and justifying.
*/
{
    char *s;
    int i, l;
    char c;

  find_break(n, &l, &c);
  if (w_len - additional + l >= width) {
    w[w_len++] = BLANK;
    words++;
    justify();
    break_line();
  }
  if (id == italic_b) {
    if (italic_flag == STOP_IT) was_italic = TRUE;
    italic_flag = START_IT;
    if (rtl) s = IT_STOP_REV; else s = IT_START;
  }
  else {
    if (italic_flag == DURING_IT) italic_flag = STOP_IT;
    else italic_flag = NO_IT;
    if (!rtl) s = IT_STOP; else s = IT_START_REV;
  }
  i = str_len(s);
  movmem(s, &w[w_len], i);
  w_len += i;
  additional += i;
}

static int process_command(int *start, int l)
{
     command_type id;
     int  i, k, k1, num_len;
     char end_char;
     command_record cmd;
     char  foot_buffer[MAX_FOOT];

  i = search_command(&r[*start+l]);
  get_command(i, &cmd);
  id = cmd.command_id;
  if (id < 0) return -1;
  if (cmd.line_command) {
    if (w_len != 0) break_line();
    *start += l + cmd.len;
  }

  if ((footnote_flag) && (id != italic_b) && (id != italic_e)) return -1;

  switch (id) {
      case center_b: centering = TRUE;  break;
      case center_e: centering = FALSE; break;

      case quote_b:
	 margin = MARGIN_SIZE + INDENT;
	 width  = LINE_SIZE - 2 * INDENT;
	 break;
      case quote_e:
	 margin = MARGIN_SIZE;
	 width  = LINE_SIZE;
	 break;

      case single_c:  spacing = 1; break;
      case double_c:  spacing = 2; break;

      case rtl_c:     rtl = TRUE;  break;
      case nortl_c:   rtl = FALSE; break;

      case newpage_c: bottom_of_page(TRUE); print_top_of_page(); break;
      case eol_c:
	 store_word(*start, l);
	 break_line();
	 n += line_length();
	 break;

      case enum_b:
	 enum_number  = 0;
	 enum_flag    = TRUE;
	 enum_written = TRUE;
	 width        = LINE_SIZE - INDENT;
	 if (!rtl) margin = MARGIN_SIZE + INDENT;
	 break;
      case enum_e:
	 margin    = MARGIN_SIZE;
	 width     = LINE_SIZE;
	 enum_flag = FALSE;
	 break;
      case item_c:
	 if (w_len != 0) break_line();
	 *start += l + cmd.len;
	 skip_blanks(start);
	 enum_written = FALSE;
	 enum_number++;
         break;

      case italic_b:
      case italic_e:
        if (id == italic_b) italic_input = TRUE;
        else if (!italic_input) return -1;
        else italic_input = FALSE;
        if (footnote_flag) {
          *start += l + cmd.len;
          return i;
        }
        store_word(*start, l);
        *start += l + cmd.len;
        italic_command(id);
        end_of_word(*start);
        break;

      case table_b:
         table_flag = TRUE;
         columns = 0;
         if (r[(*start)++] == LEFT_BRACE) end_char = RIGHT_BRACE;
         else end_char = LEFT_BRACE;
         while ((r[*start] != end_char) && (r[*start] != LF) &&
                (columns < TABLE_MAX))
           column_type[columns++] = r[(*start)++];
         break;
      case table_e:
         table_flag  = FALSE;
         break;

      case parms_c:
         k = cmd.lang;
         get_parm(&LINE_SIZE, start, k, LINE_MIN, LINE_MAX);
         get_parm(&MARGIN_SIZE, start, k, MAR_MIN, MAR_MAX);
         get_parm(&PAGE_SIZE, start, k, PAGE_MIN, PAGE_MAX);
         get_parm(&TOP, start, k, TOP_MIN, TOP_MAX);
         get_parm(&INDENT, start, k, MAR_MIN, MAR_MAX);
         get_parm(&TABLE_WIDTH, start, k, TCW_MIN, TCW_MAX);
         LINE_SIZE++;
         width  = LINE_SIZE;
         margin = MARGIN_SIZE;
         break;

      case foot_c:
	 create_foot_number(foot_buffer, &num_len);
	 movmem(&foot_buffer[0], &r[*start+l], num_len);
	     /* Ensure footnote stays with previous word */
	 store_word(*start, l+num_len);
	 *start += l + cmd.len;
	 find_end(*start, cmd.lang, &k);
         if (k >= MAX_FOOT-MAX_FOOT_NUM)
           k1 = MAX_FOOT-MAX_FOOT_NUM-1;
         else k1 = k;
         movmem(&r[*start], &foot_buffer[++num_len], k1);
	 *start += k+1;
	 end_of_word(*start);
	 save_footnote(foot_buffer, k1+num_len);
	 footnote_flag = FALSE;
	 break;

      case index_c:
	 store_word(*start, l);
	 *start += l + cmd.len;
	 find_end(*start, cmd.lang, &k);
         if (k >= STR_MAX) k1 = STR_MAX; else k1 = k;
         movmem(&r[*start], &foot_buffer[0], k1);
	 *start += k+1;
	 end_of_word(*start);
	 save_index(foot_buffer, k1, page);
	 break;
  }

  if (cmd.line_command) {
    k = line_length();
    n += k;
  }
  return id;
}

static void table_entry(int r_len)
{
     int i, j = 0;                    /* Input indices */
     int col = 0;                     /* Column counter */
     int start = 0;                   /* Start of column entry */
     int len;                         /* Length of column entry */
     int offset;                      /* Offset by column justify type */
     char c;                          /* Temporary variable */

  if (r[n] == COMMAND_CHAR) return;   /* Command is not table entry */
  w_len = INDENT / 2;                 /* Indent table a bit */
  blank_line(w);
  do {                                /* Loop over columns */
    while (r[n+start] == BLANK)       /* Skip initial blanks */
      start++;
    i = start;                        /* This is start of column entry */
    do {                              /* Search for end of entry */
      c = r[n+(i++)];
    } while ((c != COLUMN_CHAR) && (c != COMMAND_CHAR) && (c != LF) );
    j = 2;
    while (r[n+i-j] == BLANK) j++;    /* Skip trailing blanks */
    len = i - start - j + 1;
    if (len < 0) len = 0;
    if ((column_type[col] == L_C) || (column_type[col] == R_C))
      offset = (TABLE_WIDTH-len) / 2;
    else if ((column_type[col] == L_R) || (column_type[col] == R_L))
      offset = TABLE_WIDTH - len;
    else if ((column_type[col] == L_L) || (column_type[col] == R_R))
      offset = 0;
    else offset = TABLE_WIDTH;        /* Bad justification code */
    if (offset != TABLE_WIDTH) {
      if (offset < 0) offset = 0;     /* Just in case */
      movmem(&r[n+start], &w[w_len+offset], len);
    }
    w_len += TABLE_WIDTH;
    col++;
    while ((c != COLUMN_CHAR) && (c != COMMAND_CHAR) && (c != LF))
      c = r[n+(i++)];                 /* Skip to next column */
    start = i;                        /* Reset start */
  } while ((c != COMMAND_CHAR) && (c != LF) && (col < columns));
  if (w_len > width) w_len = width;
  break_line();
  n += r_len;
}

static void start_line(void)
{
     int r_len;
  n++;                                /* Skip over LF */
  r_len = line_length();
  if (table_flag) table_entry(r_len);
  else if (r[n] == BLANK) {           /* Blank first char, do not format */
    if (w_len != 0) break_line();
    movmem(&r[n], &w[0], r_len);
    n += r_len;
    w_len = r_len;
    break_line();
  }
  else if (r[n] == LF) {
    if (w_len != 0) break_line();
    break_line();
    if ( (line == PAGE_SIZE) ||         /* Orphan */
	 ((spacing == 2)  && (line == PAGE_SIZE-1))) break_line();
    right = TRUE;
  }
}

static boolean is_hebrew(char c)
/*
   Terminate english insertion at hebrew or command character.
*/
{
  return ((c >= 128)&&(c <= 153)) || (c == 39) || (c == 47) ||
	 (c == LF) || (c == ZERO) || (c == COMMAND_CHAR) ||
	 (c == LEFT_BRACE) || (c == RIGHT_BRACE) ;
}

static void copy_word(int *start)
/*
    Copy english insertion into hebrew text.
*/
{
     int l;
     char c;
  find_break(*start, &l, &c);
  store_word(*start, l);
  if (l == 0) (*start)++; else *start += l;
  skip_blanks(start);
  w[w_len++] = BLANK;
  words++;
}

static boolean check_english_insertion(void)
/*
    Insert english text in hebrew: 
      if english text overflows line, must split from end to beginning.
    Input:  h4h4h4h4 the quick brown fox jumped h3h3h3 h2h2h2 h1h1h1
                        start^<--^overflow  n^
    Output: the  quick  h3h3h3  h2h2h2 h1h1
                  h4h4h4h4 brown fox jumped
*/
{
     int l;            /* Length of english insertion */
     int l1;           /* Additional length variable */
     int overflow_len; /* Length of overflow from current row */
     int overflow;     /* End of overflow in input */
     int start;        /* Start variable for copying */
     char c;           /* Temporary variable */
     
  c = r[n];
  if (rtl &&
     ( ((c >= 'A')&&(c <= 'Z')) || ((c >= 'a')&&(c <= 'z')) ) ) {
    l = 0;
    while (!is_hebrew(r[n+l])) l++;  /* Compute len of insertion */
    overflow_len = l-(width-w_len);  /* Amount of overflow */
    if (overflow_len <= 0)           /* No overflow, return */
      return FALSE; 
    overflow = n + overflow_len;     /* Find the first break in the input, */
    find_break(overflow, &l1, &c);   /*   AFTER the overflow, i.e. */
    overflow += l1;                  /*   overflow more than minimum */
    start = overflow;
    skip_blanks(&start);
    do {
      copy_word(&start);             /* Copy from here to end of insertion */
    } while ((start - n) < l);
    justify();                       /* Justify and break full line */
    break_line();
    start = n;                       /* Reset to beginning of insertion */
    do {
      copy_word(&start);             /* Copy from here to overflow break */
    } while (start < overflow);
    n += l;
    return TRUE;
  }
  else return FALSE;   /* Not hebrew text or not english insertion */
}

static void add_word(void)
{
     int l, ok;
     char c;
  while (r[n] == LF) start_line();
  find_break(n, &l, &c);
  if ( (c == BLANK) || (c == LF) || (c == ZERO)) {
    if (!check_english_insertion()) {
      store_word(n, l);
      n += l;
      w[w_len++] = BLANK;
      words++;
    }
  }
/*  else if  n += l; */
  else {
    ok = process_command(&n, l);
    if (ok < 0) n ++;
  }
  skip_blanks(&n);
  if ((centering) && (r[n] == LF) && (ok != center_b)) break_line();
}

void frm_init(char *it_start, char *it_stop,
  int lang, boolean pg_top, boolean just)
{
    int l_start = str_len(it_start);
    int l_stop  = str_len(it_stop);
    int i;

  movmem(it_start, IT_START, l_start+1);
  movmem(it_stop,  IT_STOP,  l_stop+1);
  for (i = 0; i < l_start; i++) IT_START_REV[l_start-1-i] = IT_START[i];
  for (i = 0; i < l_stop;  i++) IT_STOP_REV[l_stop-1-i]   = IT_STOP[i];
  IT_START_REV[l_start+1] = '\0';
  IT_STOP_REV[l_stop+1]   = '\0';
  if (l_start > l_stop) italic_max = l_start; else italic_max = l_stop;
  blank_line(blanks);
  rtl           = lang;
  page_top      = pg_top;
  justify_lines = just;
  spacing       = 1;
  page          = 1;
  line          = 1;
  LINE_SIZE     = 60+1;
  MARGIN_SIZE   = 10;
  PAGE_SIZE     = 56;
  TOP           = 5;
  INDENT        = 8;
  TABLE_WIDTH   = 8;
  blank_line(w);
  w_len         = 0;
  additional    = 0;
  words         = 0;
  width         = LINE_SIZE;
  margin        = MARGIN_SIZE;
}

void frm_section(char *bf, int lpf)
{
  r       = bf;
  n       = 0;
  footnote_flag = FALSE;
  italic_input  = FALSE;
  centering     = FALSE;
  italic_flag   = NO_IT;
  was_italic    = FALSE;
  enum_flag     = FALSE;
  table_flag    = FALSE;
  print_top_of_page();
  if (r[0] == ' ') {n = -1; start_line();}
  while (r[n] != 0) add_word();
  if (w_len != 0) break_line();
  bottom_of_page(lpf);
}

void language(int *lang) {*lang = rtl;}
void reset_page(int n)   {page = n; line = 1;}
void reset_foot(void)    {foot_number = 0;}
void get_page(int *p)    {*p = page;}
void end_page(void)      {if (line != 1) bottom_of_page(1);}
